package ru.yandex.search.mail.kamaji.usertype;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.document.mail.UserTypeFrom;
import ru.yandex.search.document.mail.UserTypeThread;
import ru.yandex.search.mail.kamaji.ChangeContext;
import ru.yandex.search.mail.kamaji.Kamaji;
import ru.yandex.search.mail.kamaji.senders.Address;
import ru.yandex.search.mail.kamaji.update.AbstractFilterSearchCallback;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchResult;

public class UserTypeChecker {
    private AbstractFilterSearchCallback<?> metaCallback;

    public UserTypeChecker(final AbstractFilterSearchCallback<?> metaCallback) {
        this.metaCallback = metaCallback;
    }

    public enum CheckType {
        NONE,
        THREAD,
        FROM
    }

    public CheckType getCheck(
        final MailMetaInfo meta,
        final List<Address> addresses)
    {
        CheckType result;

        String mid = meta.get(MailMetaInfo.MID);
        String tid = meta.get(MailMetaInfo.THREAD_ID);

        if (tid != null && !tid.equalsIgnoreCase(mid)) {
            result = CheckType.THREAD;
        } else if (addresses.size() == 1) {
            result = CheckType.FROM;
        } else {
            result = CheckType.NONE;
        }

        return result;
    }

    //CSOFF: ParameterNumber
    private void searchUserType(
        final List<MailMetaInfo> metas,
        final AsyncClient searchClient,
        final String url,
        final FutureCallback<Object> callback)
        throws BadRequestException
    {
        ChangeContext context = metaCallback.context();

        StringBuilder text = new StringBuilder("url:");
        text.append(url);

        QueryConstructor searchRequest =
            new QueryConstructor("/search?IO_PRIO=" + Kamaji.IOPRIO);
        searchRequest.append(
            "prefix",
            String.valueOf(context.prefix()));
        searchRequest.append("text", text.toString());
        searchRequest.append("get", MailIndexFields.USER_TYPES);

        searchClient.execute(
            context.kamaji().config().searchConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                searchRequest.toString()),
            SearchResultConsumerFactory.OK,
            context.session().listener()
                .createContextGeneratorFor(searchClient),
            new UserTypeCallback(metas, callback));
    }
    //CSON: ParameterNumber

    public void extractUserTypes(
        final List<MailMetaInfo> threadCheck,
        final Map<String, List<MailMetaInfo>> fromCheckMap)
        throws BadRequestException
    {
        ChangeContext context = metaCallback.context();

        AsyncClient searchClient =
            context.kamaji().searchClient().adjust(
                context.session().context());

        MultiFutureCallback<Object> callback =
            new MultiFutureCallback<>(new UserTypeAggregateCallback());

        for (Map.Entry<String, List<MailMetaInfo>> entry
            : fromCheckMap.entrySet())
        {
            String email =
                MailAliases.INSTANCE.normalizeEmail(entry.getKey());

            String fromUrl = UserTypeFrom.url(
                context.prefix(),
                SearchRequestText.fullEscape(email, false));

            searchUserType(
                entry.getValue(),
                searchClient,
                fromUrl,
                callback.newCallback());
        }

        // now check threads
        for (MailMetaInfo meta: threadCheck) {
            String fromUrl = UserTypeThread.url(
                context.prefix(),
                meta.get(MailMetaInfo.THREAD_ID));

            searchUserType(
                Collections.singletonList(meta),
                searchClient,
                fromUrl,
                callback.newCallback());
        }

        callback.done();
    }

    private final class UserTypeCallback
        implements FutureCallback<SearchResult>
    {
        private final List<MailMetaInfo> metas;
        private final FutureCallback<Object> callback;

        private UserTypeCallback(
            final List<MailMetaInfo> metas,
            final FutureCallback<Object> callback)
        {
            this.metas = metas;
            this.callback = callback;
        }

        @Override
        public void completed(final SearchResult result) {
            for (int i = 0; i < metas.size(); i++) {
                MailMetaInfo meta = metas.get(i);
                if (!result.hitsArray().isEmpty()) {
                    String userType =
                        result.hitsArray().get(0).attrs()
                            .get(MailIndexFields.USER_TYPES);
                    meta.set(MailIndexFields.USER_TYPE, userType);
                }

                metaCallback.onMeta(meta);
            }

            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

    private final class UserTypeAggregateCallback
        implements FutureCallback<List<Object>>
    {
        private UserTypeAggregateCallback() {
        }

        @Override
        public void completed(final List<Object> objects) {
            metaCallback.completed();
        }

        @Override
        public void failed(final Exception e) {
            metaCallback.failed(e);
        }

        @Override
        public void cancelled() {
            metaCallback.cancelled();
        }
    }
}
