package ru.yandex.search.document.mail;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.james.mime4j.dom.address.Mailbox;

import ru.yandex.dbfields.FilterSearchFields;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.xpath.PathComponent;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.json.xpath.ValueUtils;

public class JsonMailMetaHandler implements PrimitiveHandler {
    private static final String[] ADDRESS_FIELDS = new String[] {
        OracleFields.BCC,
        OracleFields.CC,
        OracleFields.FROM,
        OracleFields.TO
    };

    private final MailMetaInfo meta;

    public JsonMailMetaHandler(final MailMetaInfo meta) {
        this.meta = meta;
    }

    private void setIfPresent(
        final JsonMap envelope,
        final String envelopeName,
        final String metaName)
        throws JsonException
    {
        String value = envelope.getString(envelopeName, "");
        if (!value.isEmpty()) {
            meta.set(metaName, value);
        }
    }

    private void processAddressField(
        final JsonMap envelope,
        final String envelopeName,
        final String metaName)
        throws JsonException
    {
        JsonList addressList = envelope.getListOrNull(envelopeName);
        if (addressList == null || addressList.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        List<Mailbox> mailboxList = new ArrayList<>(addressList.size());
        for (JsonObject addressObject: addressList) {
            JsonMap address = addressObject.asMap();
            String displayName =
                address.getString(FilterSearchFields.ADDR_DISPLAY_NAME, null);
            String local = address.getString(FilterSearchFields.ADDR_LOCAL);
            String domain =
                address.getString(FilterSearchFields.ADDR_DOMAIN, null);
            if (displayName != null && !displayName.isEmpty()) {
                sb.append('"');
                sb.append(
                    displayName.replace("\\", "\\\\").replace("\"", "\\\""));
                sb.append('"');
                sb.append(' ');
            }
            sb.append('<');
            sb.append(local);
            if (domain != null && !domain.isEmpty()) {
                sb.append('@');
                sb.append(domain);
            }
            sb.append('>');
            sb.append('\n');
            mailboxList.add(new Mailbox(displayName, local, domain));
        }
        meta.setAddressHeader(metaName, new String(sb), mailboxList);
    }

    @Override
    public void handle(final List<PathComponent> path, final Object value)
        throws JsonException
    {
        if (value != null && path.size() == 1) {
            String body = value.toString();
            String name = path.get(0).name();
            if (name != null && !body.isEmpty()) {
                switch (name) {
                    case OracleFields.BCC:
                    case OracleFields.CC:
                    case OracleFields.FROM:
                    case OracleFields.TO:
                        meta.setAddressHeader(MailMetaInfo.HDR + name, body);
                        break;

                    case OracleFields.FID:
                    case OracleFields.MID:
                    case OracleFields.RECEIVED_DATE:
                    case OracleFields.THREAD_ID:
                        meta.set(name, body);
                        break;

                    case OracleFields.IS_MIXED:
                        meta.processIsMixed(ValueUtils.asLong(value));
                        break;

                    case OracleFields.LCN:
                        long lcn = ValueUtils.asLong(value);
                        if (lcn > 0L) {
                            meta.set(MailMetaInfo.LCN, body);
                        }
                        break;

                    case OracleFields.REPLY_TO:
                        meta.setAddressHeader(
                            MailMetaInfo.REPLY_TO_FIELD,
                            body);
                        break;

                    case OracleFields.ST_ID:
                        meta.set(MailMetaInfo.STID, body);
                        break;

                    case OracleFields.SUBJECT:
                        meta.set(
                            MailMetaInfo.HDR + MailMetaInfo.SUBJECT,
                            body);
                        break;

                    case OracleFields.UNAME:
                        meta.set(MailMetaInfo.SUID, body);
                        break;
                    case OracleFields.SIZE:
                        Long mailSize = ValueUtils.asLongOrNull(value);
                        if (mailSize != null) {
                            meta.mailSize(mailSize);
                        } else {
                            meta.mailSize(-1L);
                        }
                        break;
                    default:
                        break;
                }
            }
        }
    }

    public void handle(final JsonMap envelope) throws JsonException {
        for (String field: ADDRESS_FIELDS) {
            processAddressField(envelope, field, MailMetaInfo.HDR + field);
        }
        processAddressField(
            envelope,
            "replyTo",
            MailMetaInfo.REPLY_TO_FIELD);

        setIfPresent(
            envelope,
            OracleFields.SUBJECT,
            MailMetaInfo.HDR + MailMetaInfo.SUBJECT);
        JsonMap subjectInfo =
            envelope.getMapOrNull(FilterSearchFields.SUBJECT_INFO);
        if (subjectInfo != null) {
            setIfPresent(
                subjectInfo,
                OracleFields.SUBJECT,
                MailMetaInfo.HDR + MailMetaInfo.SUBJECT
                + MailMetaInfo.NORMALIZED);
        }
        meta.set(MailMetaInfo.MID, envelope.getString(OracleFields.MID));
        meta.set(MailMetaInfo.FID, envelope.getString(OracleFields.FID));
        meta.set(
            MailMetaInfo.THREAD_ID_FIELD,
            envelope.getString(FilterSearchFields.THREAD_ID));
        meta.set(
            MailMetaInfo.RECEIVED_DATE,
            envelope.getString(FilterSearchFields.RECEIVE_DATE));
        meta.set(MailMetaInfo.STID, envelope.getString(MailMetaInfo.STID));
        setIfPresent(envelope, FilterSearchFields.TAB, MailMetaInfo.TAB);
        setIfPresent(envelope, FilterSearchFields.RFC_ID, MailMetaInfo.MSG_ID);
        JsonList labels = envelope.getListOrNull(FilterSearchFields.LABELS);
        boolean unread = true;
        if (labels != null && labels.size() > 0) {
            StringBuilder lids = new StringBuilder();
            StringBuilder lidsNames = new StringBuilder();
            JsonMap labelsInfo =
                envelope.get(FilterSearchFields.LABELS_INFO).asMap();
            for (JsonObject label: labels) {
                String lid = label.asString();
                lids.append(lid);
                lids.append('\n');
                JsonMap labelInfo = labelsInfo.get(lid).asMap();
                String labelType =
                    labelInfo.get(FilterSearchFields.TYPE)
                        .get(FilterSearchFields.TITLE).asString();

                if (FilterSearchFields.TYPE_USER.equals(labelType)) {
                    lidsNames.append(
                        labelInfo.getString(FilterSearchFields.NAME));
                    lidsNames.append('\n');
                } else {
                    JsonMap symbolicName =
                        labelInfo.getMapOrNull(
                            FilterSearchFields.SYMBOLIC_NAME);
                    if (symbolicName != null) {
                        String title = symbolicName.getString(
                            FilterSearchFields.TITLE,
                            "");
                        switch (title) {
                            case FilterSearchFields.ATTACHED_LABEL_TITLE:
                                meta.set(
                                    MailMetaInfo.HAS_ATTACHMENTS,
                                    Boolean.TRUE.toString());
                                break;
                            case FilterSearchFields.DRAFT_LABEL_TITLE:
                                meta.set(
                                    MailMetaInfo.DRAFT,
                                    Boolean.TRUE.toString());
                                break;
                            case FilterSearchFields.SEEN_LABEL_TITLE:
                                unread = false;
                                break;
                            default:
                                break;
                        }
                    } else {
                        if (FilterSearchFields.DOMAIN_LABEL.equalsIgnoreCase(
                            lid))
                        {
                            meta.set(
                                MailIndexFields.DOMAIN_LABEL,
                                labelInfo.getString(FilterSearchFields.NAME));
                        }
                    }
                }
            }
            meta.set(MailIndexFields.LIDS, new String(lids));
            if (lidsNames.length() > 0) {
                meta.set(MailIndexFields.LABELS_NAMES, new String(lidsNames));
            }
        }
        if (unread) {
            meta.set(MailIndexFields.UNREAD, Boolean.TRUE.toString());
        }
        JsonMap folder = envelope.get(FilterSearchFields.FOLDER).asMap();
        meta.set(
            MailIndexFields.FOLDER_NAME,
            folder.getString(FilterSearchFields.NAME));

        String folderType =
            folder.get(FilterSearchFields.TYPE)
                .get(FilterSearchFields.TITLE).asString();
        if (folderType.equals(FilterSearchFields.TYPE_SYSTEM)) {
            folderType = folder.get(FilterSearchFields.SYMBOLIC_NAME)
                .get(FilterSearchFields.TITLE).asString();
        }
        meta.set(MailIndexFields.FOLDER_TYPE, folderType);
        JsonList attachmentsList =
            envelope.getListOrNull(MailMetaInfo.ATTACHMENTS);
        if (attachmentsList != null) {
            List<AttachInfo> attachments =
                new ArrayList<>(attachmentsList.size());
            for (JsonObject attachment: attachmentsList) {
                attachments.add(new AttachInfo(attachment.asMap()));
            }
            meta.setAttachments(attachments);
        }
        JsonList types = envelope.getListOrNull(FilterSearchFields.TYPES);
        if (types != null && !types.isEmpty()) {
            Set<Integer> messageTypes = new HashSet<>(types.size() << 1);
            for (JsonObject typeObject: types) {
                messageTypes.add((int) typeObject.asLong());
            }
            meta.setMessageTypes(messageTypes);
        }

        meta.mailSize(envelope.getLong(FilterSearchFields.SIZE, -1L));
    }
}

