package ru.yandex.search.document.mail;

import java.io.IOException;
import java.io.StringReader;
import java.net.InetAddress;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.field.datetime.parser.DateTimeParser;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.NameValuePair;
import org.apache.james.mime4j.stream.RawBody;
import org.apache.james.mime4j.stream.RawField;
import org.apache.james.mime4j.stream.RawFieldParser;

import ru.yandex.base64.Base64Decoder;
import ru.yandex.charset.Decoder;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.StringVoidProcessor;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.QuotedPrintableDecoder;
import ru.yandex.parser.email.EmailParser;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.email.types.MessageFlags;
import ru.yandex.parser.email.types.MessageTypeToString;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.parser.mail.received.ReceivedChainParser;
import ru.yandex.parser.rfc2047.Rfc2047DecodersProvider;
import ru.yandex.parser.rfc2047.Rfc2047Parser;
import ru.yandex.util.string.StringUtils;

public class MailMetaInfo implements Rfc2047DecodersProvider {
    public static final String ATTACHMENTS = MailIndexFields.ATTACHMENTS;
    public static final String ATTACHNAME = "attachname";
    public static final String ATTACHSIZE = "attachsize_b";
    public static final String ATTACHTYPE = "attachtype";
    public static final String CONTENT_TYPE = "content_type";
    public static final String ATTACHMENTS_COUNT = "attachmentsCount";
    public static final String BCC = "bcc";
    public static final String CC = "cc";
    public static final String CONTENT_DISPOSITION = "content-disposition";
    public static final String DISPLAY_NAME = "_display_name";
    public static final String DISPOSITION_TYPE = "disposition_type";
    public static final String DOCS = "docs";
    public static final String DRAFT = "draft";
    public static final String EMAIL = "_email";
    public static final String ERROR_HID = "error_hid";
    public static final String FID = MailIndexFields.FID;
    public static final String FIRSTLINE = "firstline";
    public static final String FROM = "from";
    public static final String GATEWAY_RECEIVED_DATE = "gateway_received_date";
    public static final String HAS_ATTACHMENTS = "has_attachments";
    public static final String HDR = "hdr_";
    public static final String HEADERS = "headers";
    public static final String HEADERS_SEPARATOR = ": ";
    public static final String HID = "hid";
    public static final String HTML_BODY = "html_body";
    public static final String LCN = MailIndexFields.LCN;
    public static final String MD5 = "md5";
    public static final String MDB = "mdb";
    public static final String MESSAGE_TYPE = "message_type";
    public static final String MSG_ID = MailIndexFields.MSG_ID;
    public static final String MID = "mid";
    public static final String MIXED = "mixed";
    public static final String NORMALIZED = MailIndexFields.NORMALIZED;
    public static final String PREFIX = "prefix";
    public static final String PURE_BODY = "pure_body";
    public static final String RECEIVED = "received";
    public static final String RECEIVED_DATE = MailIndexFields.RECEIVED_DATE;
    public static final String REPLY_TO = "reply-to";
    public static final String REPLY_TO_FIELD = "reply_to";
    public static final String SENDER = "sender";
    public static final String SMTP_ID = "smtp_id";
    public static final String ALL_SMTP_IDS = "all_smtp_ids";
    public static final String ATTACH_SMTP_ID = "attach_smtp_id";
    public static final String ALL_ATTACH_SMTP_IDS = "all_attach_smtp_ids";
    public static final String RECEIVED_PARSE_ERROR = "received_parse_error";
    public static final String STID = "stid";
    public static final String SUBJECT = "subject";
    public static final String SUID = "suid";
    public static final String THREAD_ID = "thread-id";
    public static final String THREAD_ID_FIELD = MailIndexFields.THREAD_ID;
    public static final String TO = "to";
    public static final String TAB = "tab";
    public static final String UID = MailIndexFields.UID;
    public static final String URL = MailIndexFields.URL;
    public static final String X_URLS = "x_urls";
    public static final String X_YANDEX = "x-yandex-";
    public static final String X_YANDEX_FID = "x-yandex-fid";
    public static final String X_YANDEX_LABEL = "x-yandex-label";
    public static final String X_YANDEX_LOGIN = "x-yandex-login";
    public static final String X_YANDEX_MDB = "x-yandex-mdb";
    public static final String X_YANDEX_META_FIRSTLINE =
        "x-yandex-meta-firstline";
    public static final String X_YANDEX_META_HDR = "x-yandex-meta-hdr";
    public static final String X_YANDEX_META_HDRBCC = "x-yandex-meta-hdrbcc";
    public static final String X_YANDEX_META_HDRCC = "x-yandex-meta-hdrcc";
    public static final String X_YANDEX_META_HDRFROM = "x-yandex-meta-hdrfrom";
    public static final String X_YANDEX_META_HDRSUBJECT =
        "x-yandex-meta-hdrsubject";
    public static final String X_YANDEX_META_HDRTO = "x-yandex-meta-hdrto";
    public static final String X_YANDEX_META_MIXED = "x-yandex-meta-mixed";
    public static final String X_YANDEX_META_REPLYTO = "x-yandex-meta-replyto";
    public static final String X_YANDEX_MID = "x-yandex-mid";
    public static final String X_YANDEX_NOTIFY_MSG = "x-yandex-notifymsg";
    public static final String X_YANDEX_RECEIVED = "x-yandex-received";
    public static final String X_YANDEX_SSUID = "x-yandex-ssuid";
    public static final String X_YANDEX_STID = "x-yandex-stid";
    public static final String X_YANDEX_SUID = "x-yandex-suid";
    public static final String X_YANDEX_THREADID = "x-yandex-threadid";
    public static final String X_YANDEX_TIMEMARK = "x-yandex-timemark";
    public static final String X_YANDEX_UID = "x-yandex-uid";

    //filter search fields, not used by indexation
    public static final String FS_ATTACH_FILENAME = "m_fileName";
    public static final String FS_ATTACH_HID = "m_hid";

    public static final String ZERO_HID = "0";

    private static final int MESSAGE_TYPE_RADIX = 16;

    private static final long MILLIS = 1000L;

    protected final EmailParser emailParser;
    private final Map<String, String> fields = new HashMap<>();
    private final StringBuilder headers = new StringBuilder();
    private final MailMetaInfo parent;
    private final int headersLengthLimit;
    private final int headerLengthLimit;
    private final Predicate<? super InetAddress> yandexNets;
    private final HidCounter counter;
    private final StringBuilder sb;
    private final StringVoidProcessor<byte[], IOException> base64Decoder;
    private final StringVoidProcessor<byte[], IOException> qpDecoder;
    private final Decoder decoder;
    private final DecodableByteArrayOutputStream byteArray;
    private ReceivedChainParser receivedChainParser = null;
    private Set<Integer> messageTypes = null;
    private List<AttachInfo> attachments = Collections.emptyList();
    private String contentType = null;
    private String charset = null;
    private long mailSize = -1L;

    public MailMetaInfo(
        final int headersLengthLimit,
        final int headerLengthLimit,
        final Predicate<? super InetAddress> yandexNets)
    {
        parent = null;
        this.headersLengthLimit = headersLengthLimit;
        this.headerLengthLimit = headerLengthLimit;
        this.yandexNets = yandexNets;
        counter = new HidCounter();
        sb = new StringBuilder();
        base64Decoder = new StringVoidProcessor<>(new Base64Decoder());
        qpDecoder = new StringVoidProcessor<>(new QuotedPrintableDecoder());
        decoder =
            new Decoder(StandardCharsets.UTF_8, CodingErrorAction.REPLACE);
        emailParser = new EmailParser();
        byteArray = new DecodableByteArrayOutputStream();
    }

    public MailMetaInfo(final MailMetaInfo parent) {
        this.parent = parent;
        headersLengthLimit = parent.headersLengthLimit;
        headerLengthLimit = parent.headerLengthLimit;
        yandexNets = parent.yandexNets;
        counter = parent.counter;
        sb = parent.sb;
        base64Decoder = parent.base64Decoder;
        qpDecoder = parent.qpDecoder;
        decoder = parent.decoder;
        emailParser = parent.emailParser;
        byteArray = parent.byteArray;
    }

    @Nonnull
    public ReceivedChainParser receivedChainParser() {
        if (receivedChainParser == null) {
            receivedChainParser = new ReceivedChainParser(yandexNets);
        }
        return receivedChainParser;
    }

    @Override
    public StringBuilder stringBuilder(final int expectedLength) {
        sb.setLength(0);
        sb.ensureCapacity(expectedLength);
        return sb;
    }

    @Override
    public DecodableByteArrayOutputStream byteArray(final int expectedLength) {
        byteArray.reset();
        byteArray.ensureCapacity(expectedLength);
        return byteArray;
    }

    @Override
    public Decoder decoderFor(final Charset charset) {
        if (charset.equals(StandardCharsets.UTF_8)) {
            return decoder;
        } else {
            return new Decoder(charset, CodingErrorAction.REPLACE);
        }
    }

    @Override
    public StringVoidProcessor<byte[], IOException> base64Decoder() {
        return base64Decoder;
    }

    @Override
    public StringVoidProcessor<byte[], IOException> quotedPrintableDecoder() {
        return qpDecoder;
    }

    public HidCounter counter() {
        return counter;
    }

    public MailMetaInfo dropContentTypeAndCharset() {
        contentType = null;
        charset = null;
        return this;
    }

    public String contentType() {
        return contentType;
    }

    public String charset() {
        return charset;
    }

    public Map<String, String> fields() {
        String prefix = get(SUID);
        if (prefix == null) {
            prefix = get(UID);
            if (prefix == null) {
                return Collections.emptyMap();
            }
        }
        return Collections.singletonMap(PREFIX, prefix);
    }

    private void appendHeader(final String name, final String body) {
        if (headers.length() > 0) {
            headers.append('\n');
        }
        headers.append(name);
        headers.append(':');
        headers.append(' ');
        headers.append(body);
    }

    public void add(final Field field) {
        add(
            field.getName().toLowerCase(Locale.ROOT),
            field.getBody().trim(),
            field);
    }

    // CSOFF: FinalParameters
    // name expected to be lower cased
    public void add(String name, String body, final Field field) {
        if (headerLengthLimit >= 0) {
            if (name.length() > headerLengthLimit) {
                name = name.substring(0, headerLengthLimit);
            }
            if (body.length() > headerLengthLimit) {
                body = body.substring(0, headerLengthLimit);
            }
        }
        if (store(name, body, field)) {
            if (headersLengthLimit >= 0) {
                int capacityLeft = headersLengthLimit - headers.length();
                int capacityRequired = name.length() + body.length() + 2;
                if (headers.length() > 0) {
                    ++capacityRequired;
                }
                if (capacityLeft >= capacityRequired) {
                    appendHeader(name, body);
                }
            } else {
                appendHeader(name, body);
            }
        }
    }
    // CSON: FinalParameters

    public void addCgi(final String name, final String body) {
        switch (name) {
            case BCC:
            case CC:
            case FROM:
            case SUBJECT:
            case TO:
                add(X_YANDEX_META_HDR + name, body, null);
                break;

            case FID:
            case MDB:
            case MID:
            case RECEIVED:
            case STID:
            case SUID:
                add(X_YANDEX + name, body, null);
                break;

            case FIRSTLINE:
                add(X_YANDEX_META_FIRSTLINE, body, null);
                break;

            case MIXED:
                add(X_YANDEX_META_MIXED, body, null);
                break;

            case REPLY_TO:
                String decoded = decodeBase64(body);
                add(X_YANDEX_META_REPLYTO, decoded, null);
                break;

            case THREAD_ID:
                add(X_YANDEX_THREADID, body, null);
                break;

            case UID:
                set(UID, body);
                break;

            default:
                break;
        }
    }

    public boolean has(final String name) {
        return fields.containsKey(name)
            || (parent != null && parent.has(name));
    }

    private void setAttachName(final String name) {
        if (name != null) {
            set(ATTACHNAME, decode(name));
        }
    }

    private static String getParam(final RawBody body, final String name) {
        for (NameValuePair pair: body.getParams()) {
            if (pair.getName().equals(name)) {
                return pair.getValue();
            }
        }
        return null;
    }

    private String decode(final String body) {
        return Rfc2047Parser.decode(body, this);
    }

    protected String decodeBase64(final String body) {
        try {
            base64Decoder.process(body);
            base64Decoder.processWith(decoder);
        } catch (IOException e) {
            // shit goes real
            return body;
        }
        return decoder.toString();
    }

    private RawBody parse(final RawField field) {
        return RawFieldParser.DEFAULT.parseRawBody(field);
    }

    public void processIsMixed(final String body) {
        processIsMixed(Long.parseLong(body, MESSAGE_TYPE_RADIX));
    }

    public void processIsMixed(final long mixed) {
        MessageFlags flags = new MessageFlags(mixed);
        if (flags.hasAttachments()) {
            set(HAS_ATTACHMENTS, Boolean.TRUE.toString());
        }
        if (flags.isDraft()) {
            set(DRAFT, Boolean.TRUE.toString());
        }
        setMessageTypes(flags.types());
    }

    public void setMessageTypes(final Set<Integer> messageTypes) {
        this.messageTypes = messageTypes;
        sb.setLength(0);
        for (Integer type: messageTypes()) {
            sb.append(type.toString());
            sb.append(' ');
            String text = MessageTypeToString.INSTANCE.lookup(type);
            if (text != null) {
                sb.append(text);
                sb.append(' ');
            }
        }
        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
            set(MESSAGE_TYPE, sb.toString());
        }
    }

    public void setAttachments(final List<AttachInfo> attachments) {
        this.attachments = attachments;
        sb.setLength(0);
        for (AttachInfo attachment: attachments) {
            String filename = attachment.filename();
            if (filename != null && !filename.isEmpty()) {
                sb.append(filename);
                sb.append('\n');
            }
        }
        if (sb.length() > 0) {
            sb.setLength(sb.length() - 1);
            set(ATTACHMENTS, sb.toString());
        }
    }

    private static RawField field(
        final String name,
        final String body,
        final Field field)
    {
        if (field instanceof RawField) {
            return (RawField) field;
        } else {
            return new RawField(name, body);
        }
    }

    @SuppressWarnings("JdkObsolete")
    private boolean store(
        final String name,
        final String body,
        final Field field)
    {
        switch (name) {
            case BCC:
            case CC:
            case FROM:
            case TO:
            case SENDER:
                String hdrName = HDR + name;
                if (!has(hdrName)) {
                    setAddressHeader(
                        hdrName,
                        decode(body),
                        emailParser.parse(body, this));
                }
                return true;

            case CONTENT_DISPOSITION:
                RawBody rawBody = parse(field(name, body, field));
                set(DISPOSITION_TYPE, rawBody.getValue());
                setAttachName(getParam(rawBody, "filename"));
                return true;

            case "content-type":
                RawBody contentTypeBody = parse(field(name, body, field));
                String contentType = contentTypeBody.getValue();
                if (contentType != null && !contentType.isEmpty()) {
                    this.contentType = contentType.toLowerCase(Locale.ROOT);
                }
                String charset = getParam(contentTypeBody, "charset");
                if (charset != null) {
                    this.charset = charset;
                }
                if (!has(ATTACHNAME)) {
                    setAttachName(getParam(contentTypeBody, "name"));
                }
                return true;

            case RECEIVED:
                RawBody receivedBody = parse(field(name, body, field));
                if (!has(GATEWAY_RECEIVED_DATE)) {
                    List<NameValuePair> params = receivedBody.getParams();
                    if (params.size() == 1) {
                        NameValuePair pair = params.get(0);
                        if (pair.getValue() == null) {
                            long date;
                            try {
                                date =
                                    new DateTimeParser(
                                        new StringReader(pair.getName()))
                                        .parseAll()
                                        .getDate()
                                        .getTime() / MILLIS;
                            } catch (Throwable t) {
                                date = -1L;
                            }
                            if (date != -1L) {
                                set(
                                    GATEWAY_RECEIVED_DATE,
                                    Long.toString(date));
                            }
                        }
                    }
                }
                if (receivedChainParser == null) {
                    receivedChainParser = new ReceivedChainParser(yandexNets);
                }
                receivedChainParser.process(body);
                String smtpId = receivedChainParser.yandexSmtpId();
                if (smtpId != null) {
                    String allIds = StringUtils.join(
                        receivedChainParser.allYandexSmtpIds(),
                        '\n');
                    if (parent == null) {
                        set(SMTP_ID, smtpId);
                        set(ALL_SMTP_IDS, allIds);
                    } else {
                        set(ATTACH_SMTP_ID, smtpId);
                        set(ALL_ATTACH_SMTP_IDS, allIds);
                    }
                }
                ErrorInfo errorInfo = receivedChainParser.errorInfo();
                if (errorInfo != null
                    && !fields.containsKey(RECEIVED_PARSE_ERROR))
                {
                    fields.put(RECEIVED_PARSE_ERROR, errorInfo.toString());
                }
                return true;

            // What about Return-Path?
            case REPLY_TO:
                if (!has(REPLY_TO_FIELD)) {
                    setAddressHeader(
                        REPLY_TO_FIELD,
                        decode(body),
                        emailParser.parse(body, this));
                }
                return true;

            case SUBJECT:
                set(HDR + SUBJECT, decode(body));
                return true;

            case X_YANDEX_FID:
            case X_YANDEX_MID:
            case X_YANDEX_STID:
            case X_YANDEX_SUID:
                set(name.substring(X_YANDEX.length()), body);
                return false;

            case X_YANDEX_LABEL:
            case X_YANDEX_LOGIN:
            case X_YANDEX_MDB:
            case X_YANDEX_NOTIFY_MSG:
            case X_YANDEX_SSUID:
            case X_YANDEX_UID:
                return false;

            case X_YANDEX_META_HDRBCC:
            case X_YANDEX_META_HDRCC:
            case X_YANDEX_META_HDRFROM:
            case X_YANDEX_META_HDRTO:
                String decoded = decodeBase64(body);
                setAddressHeader(
                    HDR + name.substring(X_YANDEX_META_HDR.length()),
                    decoded,
                    decoder.processWith(emailParser));
                return false;

            case X_YANDEX_META_HDRSUBJECT:
                if (!body.isEmpty()) {
                    set(HDR + SUBJECT, decodeBase64(body));
                }
                return false;

            case X_YANDEX_META_MIXED:
                processIsMixed(body);
                return false;

            case X_YANDEX_RECEIVED:
                set(RECEIVED_DATE, body);
                return false;

            case X_YANDEX_META_REPLYTO:
                setAddressHeader(REPLY_TO_FIELD, body);
                return false;

            case X_YANDEX_THREADID:
                set(THREAD_ID_FIELD, body);
                return false;

            case X_YANDEX_TIMEMARK:
                if (!has(RECEIVED_DATE)) {
                    set(RECEIVED_DATE, body);
                }
                return true;

            default:
                return !name.startsWith("x-yandex-meta-");
        }
    }

    // it is assumed that name is already lower-cased
    public void set(final String name, final String value) {
        if (name.equals(HEADERS)) {
            headers.setLength(0);
            headers.append(value);
        } else {
            fields.put(name, value);
        }
    }

    public void setAddressHeader(final String name, final String value) {
        setAddressHeader(name, value, emailParser.parseDecoded(value));
    }

    public void setAddressHeader(
        final String name,
        final String value,
        final List<Mailbox> mailboxList)
    {
        if (!mailboxList.isEmpty()) {
            set(name, value);
            sb.setLength(0);
            for (Mailbox mailbox: mailboxList) {
                sb.append(mailbox.getAddress());
                sb.append('\n');
            }
            set(name + EMAIL, sb.toString());
            sb.setLength(0);
            for (Mailbox mailbox: mailboxList) {
                sb.append(
                    MailAliases.INSTANCE.normalizeEmail(mailbox.getAddress()));
                sb.append('\n');
            }
            set(name + NORMALIZED, sb.toString());
            sb.setLength(0);
            for (Mailbox mailbox: mailboxList) {
                String displayName = mailbox.getName();
                if (displayName != null) {
                    sb.append(displayName);
                    sb.append('\n');
                }
            }
            if (sb.length() > 0) {
                set(name + DISPLAY_NAME, sb.toString());
            }
        }
    }

    // it is assumed that name is already lower-cased
    public String getLocal(final String name) {
        String value = null;
        if (HEADERS.equals(name)) {
            if (headers.length() > 0) {
                value = new String(headers);
            }
        } else {
            value = fields.get(name);
        }
        return value;
    }

    private String concat(final String first, final String second) {
        sb.setLength(0);
        sb.append(first);
        sb.append('\n');
        sb.append(second);
        return sb.toString();
    }

    private String concat(final String first, final StringBuilder second) {
        sb.setLength(0);
        sb.append(first);
        sb.append('\n');
        sb.append(second);
        return sb.toString();
    }

    public String get(final String name) {
        String value = getLocal(name);
        if (parent != null) {
            String parentValue = parent.get(name);
            if (parentValue != null) {
                if (value == null) {
                    value = parentValue;
                } else {
                    value = concat(parentValue, value);
                }
            }
        }
        return value;
    }

    // TODO: Remove this after kamaji update
    public String url(final String hid) {
        String mid = get(MID);
        String url = null;
        if (mid != null) {
            String uid = get(UID);
            if (uid != null) {
                url = StringUtils.concat(uid, '_', mid, '/', hid);
            } else if (get(SUID) != null) {
                url = StringUtils.concat(mid, '/', hid);
            }
        }
        return url;
    }

    public Map<String, String> toMap() {
        Map<String, String> map;
        if (parent == null) {
            map = new HashMap<>(fields);
            if (headers.length() > 0) {
                map.put(HEADERS, new String(headers));
            }
        } else {
            map = parent.toMap();
            for (Map.Entry<String, String> entry: fields.entrySet()) {
                String name = entry.getKey();
                String value = entry.getValue();
                String parentValue = map.get(name);
                if (parentValue != null) {
                    value = concat(parentValue, value);
                }
                map.put(name, value);
            }
            if (headers.length() > 0) {
                String parentHeaders = map.get(HEADERS);
                if (parentHeaders == null) {
                    map.put(HEADERS, new String(headers));
                } else {
                    map.put(HEADERS, concat(parentHeaders, headers));
                }
            }
        }
        return map;
    }

    public Map<String, String> toFastDocMap() {
        Map<String, String> map = toMap();
        map.put(HID, ZERO_HID);
        String url = url(ZERO_HID);
        if (url != null) {
            map.put(URL, url(ZERO_HID));
        }
        return map;
    }

    public Set<Integer> messageTypes() {
        Set<Integer> result;
        if (messageTypes == null) {
            if (parent == null) {
                result = Collections.emptySet();
            } else {
                result = parent.messageTypes();
            }
        } else if (parent == null) {
            result = messageTypes;
        } else {
            Set<Integer> parentTypes = parent.messageTypes();
            if (parentTypes.isEmpty()) {
                result = messageTypes;
            } else {
                result = new TreeSet<>(parentTypes);
                result.addAll(messageTypes);
            }
        }
        return result;
    }

    public List<AttachInfo> attachments() {
        return attachments;
    }

    public long mailSize() {
        return mailSize;
    }

    public MailMetaInfo mailSize(final long mailSize) {
        this.mailSize = mailSize;
        return this;
    }

    @Nullable
    public String getSingleAddress(final String fieldName) {
        String raw = get(fieldName);
        if (raw != null) {
            int idx = raw.indexOf('\n');
            if (idx > 0) {
                raw = raw.substring(0, idx);
            } else {
                raw = null;
            }
        }
        return raw;
    }

    @Nullable
    public String getSender() {
        String from = getSingleAddress(HDR + FROM + NORMALIZED);
        if (from != null) {
            return from;
        }
        return getSingleAddress(REPLY_TO_FIELD + NORMALIZED);
    }
}

