package ru.yandex.search.document.mail;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Predicate;

import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.email.types.MessageTypeToString;
import ru.yandex.parser.query.FieldQuery;
import ru.yandex.parser.query.PredicateQueryVisitor;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.query.QueryToken;
import ru.yandex.parser.query.QuotedQuery;
import ru.yandex.parser.string.CollectionParser;

public enum MailMetaInfoPredicateParser
    implements PredicateQueryVisitor<MailMetaInfo, ParseException>
{
    ROOT {
        @Override
        public Predicate<MailMetaInfo> visit(final FieldQuery query)
            throws ParseException
        {
            List<String> fields = query.fields();
            if (fields.size() != 1) {
                throw new ParseException("Multi-field queries not allowed", 0);
            }
            Predicate<MailMetaInfo> predicate;
            QueryAtom atom = query.query();
            switch (fields.get(0)) {
                case "message_type":
                    predicate = atom.accept(MESSAGE_TYPE);
                    break;
                case "to":
                    predicate = atom.accept(TO);
                    break;
                case "to_domain":
                    predicate = atom.accept(TO_DOMAIN);
                    break;
                case "from":
                    predicate = atom.accept(FROM);
                    break;
                case "from_domain":
                    predicate = atom.accept(FROM_DOMAIN);
                    break;
                case "attach_type":
                    predicate = atom.accept(ATTACH_TYPE);
                    break;
                case "attach_ext":
                    predicate = atom.accept(ATTACH_EXT);
                    break;
                default:
                    throw new ParseException("Unknown field: " + fields, 0);
            }
            return predicate;
        }

        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query)
            throws ParseException
        {
            throw new ParseException("Bare tokens not supported", 0);
        }
    },
    MESSAGE_TYPE {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query)
            throws ParseException
        {
            String text = query.text();
            Integer boxed;
            try {
                boxed = MessageTypeToString.INSTANCE.apply(text).typeNumber();
            } catch (RuntimeException e) {
                throw new ParseException("Unknown message type: " + text, 0);
            }
            return t -> t.messageTypes().contains(boxed);
        }
    },
    TO {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String email =
                MailAliases.INSTANCE.normalizeEmail(query.text()).intern();
            return meta -> containsEmail(meta, MailMetaInfo.TO, email);
        }
    },
    TO_DOMAIN {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String domain = MailAliases.INSTANCE
                .extractAndNormalizeDomain(query.text())
                .intern();
            return meta -> containsDomain(meta, MailMetaInfo.TO, domain);
        }
    },
    FROM {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String email =
                MailAliases.INSTANCE.normalizeEmail(query.text()).intern();
            return meta -> containsEmail(meta, MailMetaInfo.FROM, email);
        }
    },
    FROM_DOMAIN {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String domain = MailAliases.INSTANCE
                .extractAndNormalizeDomain(query.text())
                .intern();
            return meta -> containsDomain(meta, MailMetaInfo.FROM, domain);
        }
    },
    ATTACH_TYPE {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String type = query.text();
            return meta -> meta.attachments().stream().anyMatch(
                attachment -> type.equals(attachment.contentType()));
        }
    },
    ATTACH_EXT {
        @Override
        public Predicate<MailMetaInfo> visit(final QueryToken query) {
            String ext = query.text();
            return meta -> meta.attachments().stream().anyMatch(
                attachment -> ext.equals(ext(attachment)));
        }
    };

    // TODO: Use tokenizer with early termination on match
    private static final CollectionParser<
        String,
        List<String>,
        RuntimeException>
            EMAIL_SPLITTER =
                new CollectionParser<>(x -> x, ArrayList::new, '\n');
    private static final CollectionParser<
        String,
        List<String>,
        RuntimeException>
            DOMAIN_SPLITTER =
                new CollectionParser<>(
                    MailAliases.INSTANCE::extractAndNormalizeDomain,
                    ArrayList::new,
                    '\n');

    public static Predicate<MailMetaInfo> parse(final String expr)
        throws ParseException
    {
        try {
            return new QueryParser(expr).parse().accept(ROOT);
        } catch (IOException e) {
            ParseException ex =
                new ParseException("Failed to parse query: " + expr, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    private static boolean containsEmail(
        final MailMetaInfo meta,
        final String field,
        final String email)
    {
        return EMAIL_SPLITTER.apply(
            meta.get(MailMetaInfo.HDR + field + MailMetaInfo.NORMALIZED))
            .contains(MailAliases.INSTANCE.normalizeEmail(email));
    }

    private static boolean containsDomain(
        final MailMetaInfo meta,
        final String field,
        final String email)
    {
        return DOMAIN_SPLITTER.apply(
            meta.get(MailMetaInfo.HDR + field + MailMetaInfo.NORMALIZED))
            .contains(MailAliases.INSTANCE.extractAndNormalizeDomain(email));
    }

    private static String ext(final AttachInfo attachment) {
        String filename = attachment.filename();
        if (filename == null) {
            return null;
        } else {
            return filename.substring(filename.lastIndexOf('.') + 1);
        }
    }

    @Override
    public Predicate<MailMetaInfo> visit(final QuotedQuery query)
        throws ParseException
    {
        throw new ParseException("Quoted expressions not allowed", 0);
    }

    @Override
    public Predicate<MailMetaInfo> visit(final FieldQuery query)
        throws ParseException
    {
        throw new ParseException("Nested fields not allowed", 0);
    }
}

