package ru.yandex.mail.search.subscriptions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.LongParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;

public class SubscriptionsStatusHandler implements HttpAsyncRequestHandler<JsonObject> {
    public static final String SUBS_STATUS = "subs_last_status";
    
    private static final CollectionParser<SubscriptionStatus, Set<SubscriptionStatus>, RuntimeException> STATUS_SET_PARSER =
        new CollectionParser<>(
            new EnumParser<>(SubscriptionStatus.class),
            LinkedHashSet::new);
    private static final CollectionParser<String, Set<String>, Exception> BACKEND_SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new,
            '\n');
    private static final CollectionParser<String, Set<String>, Exception> SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new);
    private static final CollectionParser<Long, Set<Long>, Exception> UIDS_PARSER =
        new CollectionParser<>(
            LongParser.INSTANCE,
            LinkedHashSet::new);
    private static final OptinUidsParser OPTIN_UIDS_PARSER =
        new OptinUidsParser();

    private final UniversalSearchProxy proxy;
    private final boolean pumpkin;

    public SubscriptionsStatusHandler(final UniversalSearchProxy proxy) {
        this(proxy, false);
    }

    public SubscriptionsStatusHandler(final UniversalSearchProxy proxy, final boolean pumpkin) {
        this.proxy = proxy;
        this.pumpkin = pumpkin;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            return new JsonAsyncTypesafeDomConsumer(
                entity,
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
        }
    }

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange httpAsyncExchange,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        proxy.logger().info("Handling vonidu");
        Map<String, String> emails = Collections.emptyMap();
        if (data != JsonNull.INSTANCE) {
            try {
                JsonList emailsObj = data.asMap().getList("emails");
                emails = new LinkedHashMap<>(emailsObj.size() << 1);
                for (JsonObject jobj: emailsObj) {
                    String emailRaw = jobj.asString();
                    emails.put(emailRaw, SubscriptionUtils.normalizeEmailForSubs(emailRaw));
                }
            } catch (JsonException je) {
                throw new BadRequestException(je);
            }
        }

        ProxySession session = new BasicProxySession(proxy, httpAsyncExchange, httpContext);
        SubscriptionsStatusContext context = new SubscriptionsStatusContext(session, emails);

        long mts = System.currentTimeMillis();
        long ts = TimeUnit.MILLISECONDS.toSeconds(mts);

        session.connection().setSessionInfo(
            "timestamp",
            Long.toString(ts));
        session.connection().setSessionInfo(
            "module",
            "tupita-subscriptions");
        session.connection().setSessionInfo(
            "uid",
            "0");

        if (pumpkin) {
            context.session().logger().info("Pumpkin used");
            SubscriptionsCallback callback = new SubscriptionsCallback(context);
            List<UidResult> result = new ArrayList<>(context.uids().size());
            for (Long uid: context.uids()) {
                result.add(new UidResult(uid, Collections.emptySet()));
            }
            callback.completed(result);
            return;
        }
        MultiFutureCallback<UidResult> mfcb
            = new MultiFutureCallback<>(
                new SubscriptionsCallback(context));
        for (Long uid: context.regularUids()) {
            launchSingleUid(
                new SubscriptionsSingleUserStatusContext(context, uid),
                new SingleUidCallback(mfcb.newCallback(), uid));
        }

        for (Long uid: context.subsOptinUids()) {
            session.logger().info("Launching premium uid " + uid);
            launchSinglePremiumUid(
                new SubscriptionsSingleUserStatusContext(context, uid),
                new SinglePremiumUidCallback(mfcb.newCallback(), uid));
        }
        mfcb.done();
    }

    private void launchSingleUid(
        final SubscriptionsSingleUserStatusContext context,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());

        StringBuilder text = new StringBuilder();
        text.append("url:");
        text.append('(');
        for (Map.Entry<String, String> emailEntry: context.emails().entrySet()) {
            text.append(SubscriptionUtils.urlByNormalized(context.uid, emailEntry.getValue()));
            text.append(" OR ");
        }
        text.setLength(text.length() - 4);
        text.append(')');
        text.append(" AND ");
        text.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
        text.append(":(");
        int i = 0;
        for (String type: SubscriptionsConstants.TYPES) {
            if (i != 0) {
                text.append(" OR ");
            }

            text.append(type);
            i++;
        }
        text.append(")");

        qc.append("text", text.toString());
        qc.append("service", context.user().service());
        qc.append("get", "*");
        qc.append("sort", SubscriptionsFields.SUBS_RECEIVED_DATE.stored());
        qc.append("IO_PRIO", 100);
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

        context.session().logger().info("launch Sequential request " + context.user().service() + " " + qc.toString());
        proxy.sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            100L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            callback);
    }

    private void launchSinglePremiumUid(
        final SubscriptionsSingleUserStatusContext context,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());

        StringBuilder text = new StringBuilder();
        text.append("url:");
        text.append('(');
        for (Map.Entry<String, String> emailEntry: context.emails().entrySet()) {
            text.append(
                SearchRequestText.fullEscape(
                    SubscriptionUtils.urlByNormalized(context.uid, emailEntry.getValue()),
                    false));
            text.append(" OR ");
        }
        text.setLength(text.length() - 4);
        text.append(')');
        text.append(" AND (");
        text.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
        text.append(":(");
        int i = 0;
        for (String type: SubscriptionsConstants.OPT_IN_TYPES) {
            if (i != 0) {
                text.append(" OR ");
            }

            text.append(type);
            i++;
        }
        text.append(") OR ");
        text.append(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.prefixed());
        text.append(":(");
        i = 0;
        for (String type: SubscriptionsConstants.OPT_IN_TYPES) {
            if (i != 0) {
                text.append(" OR ");
            }

            text.append(type);
            i++;
        }
        text.append(')');
        text.append(')');

        qc.append("text", text.toString());
        qc.append("service", context.user().service());
        qc.append("get", "*");
        qc.append("sort", SubscriptionsFields.SUBS_RECEIVED_DATE.stored());
        qc.append("IO_PRIO", 100);
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

        proxy.sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            100L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            callback);
    }

    private static final class SingleUidCallback
        extends AbstractFilterFutureCallback<JsonObject, UidResult> {
        private final Long uid;

        public SingleUidCallback(
            final FutureCallback<? super UidResult> callback,
            final Long uid)
        {
            super(callback);
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject backendResultObj) {
            try {
                JsonMap result = backendResultObj.asMap();
                JsonList hits = result.getList("hitsArray");

                Set<String> hidden = new LinkedHashSet<>(hits.size());
                for (JsonObject docObj: hits) {
                    JsonMap doc = docObj.asMap();
                    String email = doc.getString(SubscriptionsFields.SUBS_EMAIL.stored());
                    hidden.add(email);
                }

                callback.completed(new UidResult(uid, hidden));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static final class SinglePremiumUidCallback
        extends AbstractFilterFutureCallback<JsonObject, UidResult> {
        private final Long uid;

        public SinglePremiumUidCallback(
            final FutureCallback<? super UidResult> callback,
            final Long uid)
        {
            super(callback);
            this.uid = uid;
        }

        private boolean containsAny(final Set<String> types) {
            if (types.isEmpty()) {
                return false;
            }

            if (SubscriptionsConstants.TYPES.size() == 1) {
                return types.contains(SubscriptionsConstants.TYPES.iterator().next());
            }

            for (String type: SubscriptionsConstants.TYPES) {
                if (types.contains(type)) {
                    return true;
                }
            }

            return false;
        }

        @Override
        public void completed(final JsonObject backendResultObj) {
            try {
                JsonMap result = backendResultObj.asMap();
                JsonList hits = result.getList("hitsArray");

                Set<String> hidden = Collections.emptySet();
                Set<String> active = Collections.emptySet();
                Set<String> pending = Collections.emptySet();
                System.err.println(JsonType.HUMAN_READABLE.toString(result));
                for (JsonObject docObj: hits) {
                    JsonMap doc = docObj.asMap();
                    String email = doc.getString(SubscriptionsFields.SUBS_EMAIL.stored());
                    Set<String> hiddenTypes = doc.get(SubscriptionsFields.SUBS_HIDDEN_TYPES.stored(), Collections.emptySet(), BACKEND_SET_PARSER);
                    Set<String> activeTypes = doc.get(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.stored(), Collections.emptySet(), BACKEND_SET_PARSER);
                    if (containsAny(activeTypes)) {
                        if (active.isEmpty()) {
                            active = new LinkedHashSet<>(hits.size());
                        }
                        active.add(email);
                    } else if (containsAny(hiddenTypes)) {
                        if (hidden.isEmpty()) {
                            hidden = new LinkedHashSet<>(hits.size());
                        }
                        hidden.add(email);
                    } else {
                        if (pending.isEmpty()) {
                            pending = new LinkedHashSet<>(hits.size());
                        }
                        pending.add(email);
                    }
                }

                System.err.println("Premium finished " + hidden + " " + active + " " + pending) ;
                callback.completed(new UidResult(uid, hidden, active, pending));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private static final class UidResult {
        private final Long uid;
        private final Set<String> hidden;
        private final Set<String> active;
        private final Set<String> pending;
        private final boolean subsOptin;

        public UidResult(final Long uid, final Set<String> hidden) {
            this.uid = uid;
            this.hidden = hidden;
            this.active = Collections.emptySet();
            this.pending = Collections.emptySet();
            this.subsOptin = false;
        }

        /**
         * Subs optin constructor
         * @param uid
         * @param hidden
         * @param active
         * @param pending
         */
        public UidResult(
            final Long uid,
            final Set<String> hidden,
            final Set<String> active,
            final Set<String> pending)
        {
            this.uid = uid;
            this.hidden = hidden;
            this.active = active;
            this.pending = pending;
            this.subsOptin = true;
        }

        public SubscriptionStatus status(final String email) {
            if (hidden.contains(email)) {
                return SubscriptionStatus.HIDDEN;
            }

            if (active.contains(email)) {
                return SubscriptionStatus.ACTIVE;
            }

            if (pending.contains(email)) {
                return SubscriptionStatus.PENDING;
            }

            if (subsOptin) {
                return SubscriptionStatus.PENDING;
            } else {
                return SubscriptionStatus.ACTIVE;
            }
        }

        public Long uid() {
            return uid;
        }

        public Set<String> hidden() {
            return hidden;
        }

        public Set<String> active() {
            return active;
        }

        public Set<String> pending() {
            return pending;
        }
    }

    private final class SubscriptionsCallback
        extends AbstractProxySessionCallback<List<UidResult>>
    {
        private final SubscriptionsStatusContext context;

        public SubscriptionsCallback(
            final SubscriptionsStatusContext context)
        {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final List<UidResult> results) {
            String lastStatus = null;
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                writer.startObject();
                writer.key("subscriptions");
                writer.startArray();
                for (UidResult uidEntry: results) {
                    for (Map.Entry<String, String> emailEntry: context.emails().entrySet()) {
                        writer.startObject();
                        writer.key("uid");
                        writer.value(uidEntry.uid());
                        writer.key("email");
                        writer.value(emailEntry.getKey());
                        writer.key("status");
                        lastStatus = uidEntry.status(emailEntry.getValue()).value();
                        writer.value(lastStatus);
                        writer.endObject();
                    }
                }
                writer.endArray();
                writer.endObject();

                context.session().connection().setSessionInfo(
                    SUBS_STATUS,
                    lastStatus);

                context.session().connection().setSessionInfo(
                    "result",
                    sbw.toString());

                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(context.session().acceptedCharset())));
            } catch (IOException e) {
                failed(e);
            }
        }
    }

    private final class SubscriptionsStatusContext {
        private final AsyncClient client;
        private final ProxySession session;
        private final JsonType jsonType;
        // pair of raw <-> normalized emails
        private final Map<String, String> emails;
        private final Set<Long> allUids;
        private final Set<Long> regularUids;
        private final Set<Long> subsOptinUids;

        public SubscriptionsStatusContext(
            final ProxySession session,
            final Map<String, String> emails)
            throws BadRequestException
        {
            allUids = session.params().get("uid", UIDS_PARSER);
            subsOptinUids = session.params().get("opt_in_subs_uid", Collections.emptySet(), OPTIN_UIDS_PARSER);
            this.regularUids = new LinkedHashSet<>(allUids);
            this.regularUids.removeAll(subsOptinUids);
            this.session = session;
            CgiParams params = session.params();
            jsonType = JsonTypeExtractor.NORMAL.extract(params);

            if (emails == null || emails.isEmpty()) {
                Set<String> emailsRaw = params.get("email", SET_PARSER);
                this.emails = new LinkedHashMap<>(emailsRaw.size() << 1);
                for (String emailRaw: emailsRaw) {
                    this.emails.put(emailRaw, SubscriptionUtils.normalizeEmailForSubs(emailRaw));
                }
            } else {
                this.emails = emails;
            }

            if (this.emails.isEmpty()) {
                throw new BadRequestException("No emails were supplied");
            }

            client = proxy.searchClient().adjust(session.context());
        }

        public AsyncClient client() {
            return client;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Map<String, String> emails() {
            return emails;
        }

        public Set<Long> regularUids() {
            return regularUids;
        }

        public Set<Long> subsOptinUids() {
            return subsOptinUids;
        }

        public Set<Long> uids() {
            return allUids;
        }
    }

    private static final class SubscriptionsSingleUserStatusContext
        implements UniversalSearchProxyRequestContext
    {
        private final long uid;
        private final User user;
        private final SubscriptionsStatusContext context;

        public SubscriptionsSingleUserStatusContext(
            final SubscriptionsStatusContext context,
            final Long uid)
        {
            this.context = context;
            this.uid = uid;
            user = new User(SubscriptionsConstants.SUBSCRIPTIONS_SERVICE, new LongPrefix(uid));
        }

        public Map<String, String> emails() {
            return context.emails();
        }

        public ProxySession session() {
            return context.session();
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return context.client();
        }

        @Override
        public Logger logger() {
            return context.session().logger();
        }

        @Override
        public long lagTolerance() {
            return 0L;
        }
    }

    private static class OptinUidsParser
        extends CollectionParser<Long, Set<Long>, Exception>
    {
        public OptinUidsParser() {
            super(LongParser.INSTANCE, LinkedHashSet::new);
        }

        @Override
        public Set<Long> process(final char[] buf, final int off, final int len) throws Exception {
            if (len == 0) {
                return Collections.emptySet();
            }

            return super.process(buf, off, len);
        }
    }
}
