package ru.yandex.search.migrations_worker;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;

import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.AsyncHttpServerBase;
import ru.yandex.msearch.proxy.config.ImmutableAsyncHttpServerBaseConfig;

public class MigrationsWorkerLock {
    private static final int LOCK_TIMEOUT = 10000;
    private static final int REFRESH_TIMEOUT = LOCK_TIMEOUT / 2;
    private static final String LOCK_URL =
            "/_producer_lock?service=change_log&session-timeout=" + LOCK_TIMEOUT + "&producer-name=migrations_worker";
    private static final String REFRESH_URL_TEMPLATE =
            "/_refresh_lock?service=change_log&lockid=%s";

    private final AsyncHttpServerBase
        <? extends ImmutableAsyncHttpServerBaseConfig> service;
    private final PrefixedLogger logger;

    private String lockId = null;
    private long expireTs = -1;

    public MigrationsWorkerLock(
        final AsyncHttpServerBase<? extends ImmutableAsyncHttpServerBaseConfig>
            service)
    {
        this.service = service;
        this.logger = service.logger().replacePrefix(MigrationsWorkerLock.class.getSimpleName());
    }

    public boolean tryLock() {
        if (isLocked()) {
            return (System.currentTimeMillis() + REFRESH_TIMEOUT < expireTs) || refreshLock();
        } else {
            return lock();
        }
    }

    private boolean lock() {
        try {
            logger.info("Locking");
            Future<HttpResponse> future =
                    service.producerClient().execute(
                            service.config().producerClientConfig().host(),
                            new BasicAsyncRequestProducerGenerator(LOCK_URL),
                            EmptyFutureCallback.INSTANCE);
            logger.info("Waiting for future");
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_OK) {
                String lockId = CharsetUtils.toString(response.getEntity());
                logger.info("Locked with " + lockId);
                this.lockId = lockId;
                this.expireTs = System.currentTimeMillis() + LOCK_TIMEOUT;
                return true;
            } else {
                logger.info("Failed to obtain lock, current is " + CharsetUtils.toString(response.getEntity()));
            }
        } catch (Exception e) {
            logger.info("Failed to lock: " + e);
        }
        return false;
    }

    private boolean refreshLock() {
        Future<HttpResponse> future =
                service.producerClient().execute(
                        service.config().producerClientConfig().host(),
                        new BasicAsyncRequestProducerGenerator(String.format(REFRESH_URL_TEMPLATE, lockId)),
                        EmptyFutureCallback.INSTANCE);
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_OK) {
                this.expireTs = System.currentTimeMillis() + LOCK_TIMEOUT;
                return true;
            } else {
                logger.fine("Failed to refresh lock");
            }
        } catch (InterruptedException | ExecutionException e) {
            logger.log(Level.WARNING, "Failed to get lock", e);
        }
        return false;
    }

    public boolean isLocked() {
        return lockId != null && System.currentTimeMillis() < expireTs;
    }

}
