package ru.yandex.search.migrations_worker;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.MigrationsTasksPostgresDao;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.pojo.MigrationTaskStatus;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class MigrationsWorkerStater implements Stater {
    private static final long UPDATE_PERIOD_MS = 5000;
    private static final long PERCENTILES_PERIOD_MS = TimeUnit.MINUTES.toMillis(5);
    private static final List<Integer> PERCENTILES = List.of(5, 25, 50, 75, 90, 95, 99);

    private final MigrationsTasksPostgresDao migrationsTasksPostgresDao;
    private final TasksPlanner tasksPlanner;

    private MigrationsTasksPostgresDao.Stat taskStats = null;
    private long lastUpdate = 0L;

    public MigrationsWorkerStater(MigrationsTasksPostgresDao migrationsTasksPostgresDao, TasksPlanner tasksPlanner) {
        this.migrationsTasksPostgresDao = migrationsTasksPostgresDao;
        this.tasksPlanner = tasksPlanner;
    }

    @Override
    public <E extends Exception> void stats(StatsConsumer<? extends E> statsConsumer) throws E {
        if (lastUpdate + UPDATE_PERIOD_MS < System.currentTimeMillis()) {
            taskStats = migrationsTasksPostgresDao.getStats(PERCENTILES, PERCENTILES_PERIOD_MS);
            lastUpdate = System.currentTimeMillis();
        }

        for (Map.Entry<MigrationTaskStatus, Integer> entry : taskStats.getTaskCounts().entrySet()) {
            statsConsumer.stat(
                    "tasks-count-" + entry.getKey().value() + "_axxx",
                    entry.getValue()
            );
        }

        for (Map.Entry<Integer, Long> entry: taskStats.getTimings().entrySet()) {
            statsConsumer.stat(
                    "task-processing-time-" + String.format("%02d", entry.getKey()) + "-perc_axxx",
                    entry.getValue()
            );
        }

        statsConsumer.stat("running-planners_axxx", tasksPlanner.isRunning() ? 1 : 0);
    }

    @Override
    public void addToGolovanPanel(GolovanPanel panel, String statsPrefix) {
        ImmutableGolovanPanelConfig config = panel.config();
        GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

        GolovanChart instancesChart = new GolovanChart(
                "number",
                " instances",
                false,
                false,
                0d);
        instancesChart.addSplitSignal(
                config,
                statsPrefix + "running-planners_axxx",
                0,
                true,
                false);
        group.addChart(instancesChart);

        GolovanChart timingsChart = new GolovanChart(
                "timings",
                " ms",
                false,
                false,
                0d
        );
        for (int perc : PERCENTILES) {
            timingsChart.addSignal(new GolovanSignal(
                    statsPrefix + "task-processing-time-" + String.format("%02d", perc) + "-perc_axxx",
                    config.tag(),
                    String.format("0.%02d", perc),
                    null,
                    0,
                    false
            ));
        }
        group.addChart(timingsChart);

        for (MigrationTaskStatus status : MigrationTaskStatus.values()) {
            GolovanChart countsChart = new GolovanChart(
                    status.value(),
                    " tasks-number",
                    false,
                    false,
                    0d);
            countsChart.addSplitSignal(
                    config,
                    statsPrefix + "tasks-count-" + status.value() + "_axxx",
                    0,
                    false,
                    false);
            group.addChart(countsChart);
        }

        panel.addCharts("migrations", "Migrations ", group);
    }

    @Override
    public void addToAlertsConfig(
            @Nonnull IniConfig alertsConfig,
            @Nonnull ImmutableGolovanPanelConfig panelConfig,
            @Nonnull String statsPrefix)
            throws BadRequestException
    {
        ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
        alerts.createAlert(
                alertsConfig,
                alerts.module() + "-running-planers-min",
                statsPrefix + "running-planners_axxx",
                new AlertThresholds(1.5, null, null) // more than 1
        );
        alerts.createAlert(
                alertsConfig,
                alerts.module() + "-running-planers-max",
                statsPrefix + "running-planners_axxx",
                new AlertThresholds(null, 0.5, null) // less than 1
        );
        alerts.createAlert(
                alertsConfig,
                alerts.module() + "-tasks-created",
                statsPrefix + "tasks-count-" + MigrationTaskStatus.CREATED.value() + "_axxx",
                new AlertThresholds(100., null, null)
        );
        alerts.createAlert(
                alertsConfig,
                alerts.module() + "-tasks-failed",
                statsPrefix + "tasks-count-" + MigrationTaskStatus.FAILED.value() + "_axxx",
                new AlertThresholds(5., null, null)
        );
    }
}
