package ru.yandex.search.migrations_worker;

import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.mail.MailSearchServices;
import ru.yandex.msearch.proxy.AsyncHttpServerBase;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.FuritaBatchingMover;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.SequentialMover;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.SubscriptionsUpdateContext;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.SubscriptionsUpdateStatusHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.UpdateAction;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.UpdateItem;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.MigrationsTasksPostgresDao;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.pojo.MigrationTask;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.pojo.MigrationTaskStatus;
import ru.yandex.msearch.proxy.config.ImmutableAsyncHttpServerBaseConfig;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;

public class TaskRunner implements Runnable {
    private final static int BATCH_SIZE = 200;

    private final AsyncHttpServerBase
        <? extends ImmutableAsyncHttpServerBaseConfig> service;
    private final MigrationsTasksPostgresDao tasksDao;
    private final MigrationTask task;
    private final PrefixedLogger logger;

    public TaskRunner(
            AsyncHttpServerBase<? extends ImmutableAsyncHttpServerBaseConfig>
                service,
            MigrationsTasksPostgresDao tasksDao,
            MigrationTask task,
            PrefixedLogger logger)
    {
        this.service = service;
        this.tasksDao = tasksDao;
        this.task = task;
        this.logger = logger.replacePrefix(task.getRequestId());
    }

    @Override
    public void run() {
        logger.info("Running task");
        SubscriptionsUpdateContext context = buildContext();
        ProcessingFinishedCallback processingFinishedCallback = new ProcessingFinishedCallback();
        if (context.optIn()) {
            logger.info("Opt-in enabled");
            OptinStoreCompletedCallback optinStoreCompletedCallback =
                    new OptinStoreCompletedCallback(
                            processingFinishedCallback,
                            context,
                            List.of(task.toUpdateItem(context))
                    );

            QueryConstructor foldersQuery = new QueryConstructor(
                    new StringBuilder(service.config().foldersConfig().uri().toASCIIString())
                            .append(service.config().foldersConfig().firstCgiSeparator())
                            .append(ProxyParams.WMI_SUFFIX));
            try {
                foldersQuery.append("mdb", "pg");
            } catch (BadRequestException e) {
                throw new RuntimeException(e);
            }
            foldersQuery.append("uid", context.uid());

            try {
                service.foldersClient().execute(
                        new HeaderAsyncRequestProducerSupplier(
                                new AsyncGetURIRequestProducerSupplier(foldersQuery.toString()),
                                new BasicHeader(
                                        YandexHeaders.X_YA_SERVICE_TICKET,
                                        service.filterSearchTvm2Ticket()
                                )),
                        JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                        context.listener().createContextGeneratorFor(service.foldersClient()),
                        optinStoreCompletedCallback).get();
            } catch (URISyntaxException | InterruptedException | ExecutionException use) {
                processingFinishedCallback.failed(use);
            }
        } else {
            logger.info("Simple updates finished, launching with flag move existing");
            if (!context.optIn() || task.getAction() != UpdateAction.ACTIVATE) {
                logger.info("Update action not activate or not optin, use simple mover " + task.getAction());
                SequentialMover mover = new SequentialMover(
                        context,
                        service,
                        processingFinishedCallback,
                        List.of(task.toUpdateItem(context))
                );
                mover.launchFirst();
            } else {
                logger.info("Optin and activate action: use furita rules mover");
                FuritaBatchingMover mover = new FuritaBatchingMover(
                        service,
                        context,
                        List.of(task.toUpdateItem(context)),
                        processingFinishedCallback
                );
                mover.next(0);
            }
        }
    }

    private SubscriptionsUpdateContext buildContext() {
        BasicHttpContext context = new BasicHttpContext();
        context.setAttribute(
                HttpCoreContext.HTTP_REQUEST,
                new BasicHttpRequest("", "")
        );
        return new SubscriptionsUpdateContext(
                task.getUid(),
                new User(MailSearchServices.CHANGE_LOG.service(), new LongPrefix(task.getUid())),
                service.searchClient(),
                service.producerStoreClient(),
                service.mopsClient(),
                new BasicRequestsListener(),
                logger,
                context,
                task.getRequestId(),
                false,
                BATCH_SIZE,
                task.isOptIn()
        );
    }

    private class ProcessingFinishedCallback implements FutureCallback<Object> {
        @Override
        public void completed(Object o) {
            tasksDao.updateStatus(task, MigrationTaskStatus.FINISHED);
            logger.info("Callback completed for task");
        }

        @Override
        public void failed(Exception e) {
            logger.log(Level.INFO, "Callback failed for task: ", e);
            tasksDao.updateStatus(task, MigrationTaskStatus.FAILED, task.getRetryCount() + 1);
        }

        @Override
        public void cancelled() {
            logger.info("Callback cancelled for task: ");
        }
    }

    private class OptinStoreCompletedCallback
            extends AbstractFilterFutureCallback<JsonObject, Object>
    {
        private final SubscriptionsUpdateContext context;
        private final List<UpdateItem> items;

        public OptinStoreCompletedCallback(
                final FutureCallback<? super Object> callback,
                final SubscriptionsUpdateContext context,
                final List<UpdateItem> items)
        {
            super(callback);
            this.context = context;
            this.items = items;
        }

        @Override
        public void completed(final JsonObject result) {
            String pendingFid = null;
            String inboxFid = null;
            try {
                JsonMap foldersMap = result.get("folders").asMap();

                for (Map.Entry<String, JsonObject> folder : foldersMap.entrySet()) {
                    JsonMap map = folder.getValue().asMap();
                    JsonObject fType = map.get("type");
                    if (fType.type() != JsonObject.Type.MAP) {
                        continue;
                    }

                    String typeTitle = fType.get("title").asStringOrNull();

                    String fid = folder.getKey();
                    JsonMap symbolicNameMap = map.getMapOrNull("symbolicName");
                    if (symbolicNameMap == null) {
                        continue;
                    }

                    String symbolicName = symbolicNameMap.getString("title");


                    if ("system".equalsIgnoreCase(typeTitle)) {
                        if (SubscriptionsUpdateStatusHandler.OPT_IN_PENDING_FOLDER.equals(symbolicName)) {
                            if (pendingFid != null) {
                                context.logger().warning(
                                        "More than one pending folder " + pendingFid + " " + fid);
                                continue;
                            }
                            pendingFid = fid;
                        }

                        if ("inbox".equals(symbolicName)) {
                            if (inboxFid != null) {
                                context.logger().warning(
                                        "More than one inbox folder " + inboxFid + " " + fid);
                                continue;
                            }
                            inboxFid = fid;
                        }
                    }
                }
            } catch (JsonException je) {
                context.logger().info(
                        "Failed to parse folders json "
                                + JsonType.HUMAN_READABLE.toString(result));
                callback.failed(je);
                return;
            }

            context.pendingFid(pendingFid);
            context.inboxFid(inboxFid);
            context.logger().info(
                    "Simple updates finished, launching with flag move existing");
            UpdateAction updateAction = items.get(0).action();
            if (!context.optIn() || updateAction != UpdateAction.ACTIVATE) {
                context.logger().info(
                        "Update action not activate or not optin, use simple mover " + updateAction);
                SequentialMover mover = new SequentialMover(context, service, callback, items);
                mover.launchFirst();
            } else {
                context.logger().info(
                        "Optin and activate action: use furita rules mover");
                FuritaBatchingMover mover = new FuritaBatchingMover(
                        service,
                        context,
                        items,
                        callback
                );
                mover.next(0);
            }
        }
    }
}
