package ru.yandex.search.msal;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.server.sync.HttpSession;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.msal.pool.DBConnectionPool;
import ru.yandex.search.msal.processors.TimestampFieldProcessor;

public class GetMinTransactionDateHandler extends AbstractMasterHandler {
    private static final String DEFAULT_VALUE =
        Integer.toString(Integer.MAX_VALUE);

    public GetMinTransactionDateHandler() {
        super(false);
    }

    private static String getTimestamp(final PreparedStatement stmt)
        throws HttpException, SQLException
    {
        try (ResultSet resultSet = stmt.executeQuery()) {
            if (!resultSet.next()) {
                throw new ServiceUnavailableException();
            }
            if (resultSet.getBoolean(1)) {
                throw new ServiceUnavailableException("This is a replica");
            }
            Timestamp timestamp = resultSet.getTimestamp(2);
            if (timestamp == null) {
                return null;
            } else {
                return TimestampFieldProcessor.timestampToString(timestamp);
            }
        }
    }

    @Override
    public void handlePgRequest(
        final HttpSession session,
        final DBConnectionPool connPool,
        final long uid)
        throws HttpException, IOException, SQLException
    {
        Logger logger = session.logger();
        try (Connection connection = connPool.getConnection(logger);
            PreparedStatement stmt = connection.prepareStatement(
                "select pg_is_in_recovery(), code.min_xact()"))
        {
            String minDate = getTimestamp(stmt);
            logger.info(
                "Transaction date retrieved from database: " + minDate);
            JsonType type = JsonTypeExtractor.NULL.extract(session.params());
            if (type == null) {
                if (minDate == null) {
                    minDate = DEFAULT_VALUE;
                }
                session.response().setEntity(
                    new StringEntity(
                        minDate,
                        CharsetUtils.acceptedCharset(session.request())));
            } else {
                String serverTimestamp;
                try (PreparedStatement stmtNow = connection.prepareStatement(
                        "select pg_is_in_recovery(), now()"))
                {
                    serverTimestamp = getTimestamp(stmtNow);
                }
                logger.info("Server timestamp: " + serverTimestamp);
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = type.create(sbw)) {
                    writer.startObject();
                    writer.key("min_transaction_date");
                    writer.value(minDate);
                    writer.key("server_timestamp");
                    writer.value(serverTimestamp);
                    writer.endObject();
                }
                session.response().setEntity(
                    new StringEntity(
                        sbw.toString(),
                        CharsetUtils.acceptedCharset(session.request())));
            }
        }
    }

    @Override
    public String toString() {
        return "Retrieves start_date of first of active transactions "
            + "or Integer.MAX_VALUE";
    }
}

