package ru.yandex.search.msal;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;

import org.apache.http.HttpException;

import ru.yandex.http.server.sync.HttpSession;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.msal.pool.DBConnectionPool;
import ru.yandex.search.sharpei.ConnInfo;
import ru.yandex.search.sharpei.ShardInfo;
import ru.yandex.search.sharpei.SharpeiClient;
import ru.yandex.search.sharpei.SharpeiMode;

public class MailHandler implements ScopeHandler {
    private static final String NO_RO = "no-ro";
    private static final String UID = "uid";

    private final ConcurrentHashMap<String, DBConnectionPool> mailPools =
        new ConcurrentHashMap<>();
    private final SharpeiClient sharpeiClient;
    private final Config config;
    private final Server server;

    public MailHandler(final Server server) {
        this.server = server;

        this.config = server.config();
        this.sharpeiClient = server.sharpeiClient();
    }

    @Override
    public void handle(
        final HttpSession session,
        final DatabaseHandler handler)
        throws HttpException, IOException, SQLException
    {
        RequestsListener listener = new BasicRequestsListener();
        session.context().getConnection(LoggingServerConnection.class)
            .setUpstreamStats(listener);
        CgiParams params = session.params();

        boolean pgMaster =
            params.getBoolean(NO_RO, false)
                || !handler.pgReplicasAllowed();
        String pgShard = params.getString("pgshard", null);
        if (pgShard == null) {
            Long uid = params.getLong(UID);
            SharpeiMode mode;
            if (pgMaster) {
                mode = SharpeiMode.WRITE_ONLY;
            } else {
                mode = SharpeiMode.READ_WRITE;
            }
            ConnInfo connInfo;
            try {
                connInfo = sharpeiClient.connInfo(
                    uid,
                    mode,
                    listener.createContextGeneratorFor(sharpeiClient),
                    EmptyFutureCallback.INSTANCE)
                    .get();
            } catch (ExecutionException e) {
                throw HttpExceptionConverter.toHttpException(e.getCause());
            } catch (InterruptedException e) {
                throw new ServiceUnavailableException(e);
            }
            session.response().addHeader(
                YandexHeaders.PGSHARD,
                connInfo.shardId());
            handler.handlePgRequest(
                session,
                server.pool(
                    connInfo,
                    mailPools,
                    config.pgPoolConfig(),
                    session.logger()),
                uid);
        } else {
            ConnInfo connInfo = server.connInfoCache().getIfPresent(pgShard);
            if (pgMaster || connInfo == null) {
                connInfo = null;
                ShardInfo shardInfo;
                try {
                    shardInfo = sharpeiClient.shardInfo(
                        pgShard,
                        listener.createContextGeneratorFor(sharpeiClient),
                        EmptyFutureCallback.INSTANCE)
                        .get();
                } catch (ExecutionException e) {
                    throw HttpExceptionConverter.toHttpException(e.getCause());
                } catch (InterruptedException e) {
                    throw new ServiceUnavailableException(e);
                }

                if (pgMaster || shardInfo.replicas().isEmpty()) {
                    connInfo = shardInfo.master();
                } else {
                    connInfo = shardInfo.replicas().get(0);
                    if (connInfo != null) {
                        session.logger().info("Putting conninfo in cache for " + pgShard);
                        server.connInfoCache().put(pgShard, connInfo);
                    }
                }
            } else {
                session.logger().info("Got conninfo from cache for " + pgShard);
            }

            if (connInfo == null) {
                throw new NotFoundException(
                    "Can't locate shard " + pgShard);
            }
            long uid;
            if (handler.uidRequired()) {
                uid = params.getLong(UID);
            } else {
                uid = -1L;
            }
            handler.handlePgRequest(
                session,
                server.pool(
                    connInfo,
                    mailPools,
                    config.pgPoolConfig(),
                    session.logger()),
                uid);
        }
    }

    @Override
    public Map<String, Object> status(
        final Map<String, Object> status,
        final boolean verbose)
    {
        if (verbose) {
            status.put("pg-pools-status", new TreeMap<>(mailPools));
        }

        return status;
    }
}
