package ru.yandex.search.msal;

import java.io.IOException;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;

import ru.yandex.collection.Pattern;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.server.sync.HttpSession;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.GatewayTimeoutException;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.msal.contacts.ContactsHandler;
import ru.yandex.search.msal.contacts.GetContactsHandler;
import ru.yandex.search.msal.contacts.GetEmailsHandler;
import ru.yandex.search.msal.contacts.GetListInfoHandler;
import ru.yandex.search.msal.contacts.GetSharedByClientHandler;
import ru.yandex.search.msal.contacts.GetSharedByOwnerContactsHandler;
import ru.yandex.search.msal.contacts.GetTaggedContacts;
import ru.yandex.search.msal.contacts.GetTaggedEmails;
import ru.yandex.search.msal.contacts.GetTagsHandler;
import ru.yandex.search.msal.pool.DBConnectionPool;
import ru.yandex.search.msal.pool.ImmutablePoolConfig;
import ru.yandex.search.sharpei.ConnInfo;
import ru.yandex.search.sharpei.SharpeiClient;

public class Server extends BaseHttpServer<Config> {
    private static final int CACHE_MINUTES_EXPIRE = 2;
    private static final int CACHE_SIZE = 10000;
    public static final int ACCESS_DENIED = 22;
    public static final int SESSION_KILLED = 28;
    public static final int INVALID_CHARACTER = 911;
    public static final int INVALID_NUMBER = 1722;
    public static final int INVALID_MDB = 12154;
    public static final int DATABASE_TIMEOUT = 12170;
    public static final int VALUE_TOO_LARGE_FOR_COLUMN = 12899;

    private static final String POOL_CREATED = "Pool created for ";

    private final SharedConnectingIOReactor reactor;
    private final SharpeiClient sharpeiClient;
    private final SharpeiClient orgSharpeiClient;
    private final MailHandler mailHandler;
    private final ContactsHandler contactsHandler;
    private final Cache<String, ConnInfo> connInfoCache;

    public Server(final Config config) throws IOException {
        super(config);
        reactor = new SharedConnectingIOReactor(
            config,
            config.dnsConfig(),
            new ThreadGroup(getThreadGroup(), config.name() + "-Client"));
        sharpeiClient = new SharpeiClient(reactor, config.sharpeiConfig());
        orgSharpeiClient =
            new SharpeiClient(reactor, config.orgSharpeiConfig());
        this.mailHandler = new MailHandler(this);
        this.contactsHandler = new ContactsHandler(this);

        registerStaters(config().orgSharpeiStaters());

        connInfoCache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_MINUTES_EXPIRE, TimeUnit.MINUTES)
            .maximumSize(CACHE_SIZE)
            .concurrencyLevel(config().workers()).build();
        register(
            new Pattern<>("/operations-queue-envelopes", false),
            new DatabaseHandlerAdapter(this, new ScopeOperationQueueHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/set-firstline", false),
            new DatabaseHandlerAdapter(this, new SetFirstlineHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-firstline", false),
            new DatabaseHandlerAdapter(this, new GetFirstlineHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-min-transaction-date", false),
            new DatabaseHandlerAdapter(
                this,
                new GetMinTransactionDateHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-user-revision", false),
            new DatabaseHandlerAdapter(this, new GetUserRevisionHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/user-mids", false),
            new DatabaseHandlerAdapter(this, new UserMidsHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-mid-by-message-id", false),
            new DatabaseHandlerAdapter(this, new GetMidByMessageIdHandler()),
            RequestHandlerMapper.GET);

        // contacts
        register(
            new Pattern<>("/get-user-contacts", false),
            new DatabaseHandlerAdapter(this, new GetContactsHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-user-emails", false),
            new DatabaseHandlerAdapter(this, new GetEmailsHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-contacts-list", false),
            new DatabaseHandlerAdapter(this, new GetListInfoHandler()),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/get-contacts-tags", false),
            new DatabaseHandlerAdapter(this, new GetTagsHandler()),
            RequestHandlerMapper.GET);

        register(
                new Pattern<>("/get-tagged-contacts", false),
                new DatabaseHandlerAdapter(this, new GetTaggedContacts()),
                RequestHandlerMapper.GET);
        register(
                new Pattern<>("/get-tagged-emails", false),
                new DatabaseHandlerAdapter(this, new GetTaggedEmails()),
                RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-shared-lists", false),
            new DatabaseHandlerAdapter(this, new GetSharedByOwnerContactsHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/get-shared-owners-lists", false),
            new DatabaseHandlerAdapter(this, new GetSharedByClientHandler()),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/list-shards", false),
            new ListShardsHandler(this),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("/get-user-shard", false),
            new GetUserShardHandler(this),
            RequestHandlerMapper.GET);
    }

    @Override
    public void start() throws IOException {
        reactor.start();
        sharpeiClient.start();
        orgSharpeiClient.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        super.close();
        sharpeiClient.close();
        orgSharpeiClient.close();
        reactor.close();
    }

    public SharpeiClient sharpeiClient() {
        return sharpeiClient;
    }

    public SharpeiClient orgSharpeiClient() {
        return orgSharpeiClient;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("sharpei", sharpeiClient.status(verbose));
        status.put("sharpei-connect", orgSharpeiClient.status(verbose));
        mailHandler.status(status, verbose);
        contactsHandler.status(status, verbose);
        return status;
    }

    // CSOFF: ParameterNumber
    public DBConnectionPool pool(
        final ConnInfo connInfo,
        final ConcurrentHashMap<String, DBConnectionPool> pools,
        final ImmutablePoolConfig poolConfig,
        final Logger logger)
    {
        String url = connInfo.pgUrl();
        DBConnectionPool pool = pools.get(url);
        if (pool == null) {
            DBConnectionPool newPool =
                new DBConnectionPool(url, poolConfig, logger);
            pool = pools.putIfAbsent(url, newPool);
            if (pool == null) {
                pool = newPool;
                logger.info(POOL_CREATED + config.name() + ' ' + url);
            }
        }

        return pool;
    }
    // CSON: ParameterNumber

    public void handleException(
        final SQLException e,
        final Logger logger)
        throws HttpException
    {
        String message = "Database responded with code " + e.getErrorCode()
            + " and state " + e.getSQLState();
        logger.log(Level.WARNING, "SQL request failed", e);
        switch (e.getErrorCode()) {
            case INVALID_CHARACTER:
            case INVALID_NUMBER:
            case INVALID_MDB:
            case VALUE_TOO_LARGE_FOR_COLUMN:
                throw new BadRequestException(message, e);
            case DATABASE_TIMEOUT:
                throw new GatewayTimeoutException(message, e);
            case ACCESS_DENIED:
            case SESSION_KILLED:
            default:
                throw new BadGatewayException(message, e);
        }
    }

    // CSOFF: MethodLength
    public void handle(
        final HttpSession session,
        final DatabaseHandler handler)
        throws HttpException, IOException, SQLException
    {
        RequestsListener listener = new BasicRequestsListener();
        session.context().getConnection(LoggingServerConnection.class)
            .setUpstreamStats(listener);
        CgiParams params = session.params();
        MsalScope scope =
            params.getEnum(MsalScope.class, "scope", MsalScope.MAIL);
        if (scope == MsalScope.CONTACTS) {
            contactsHandler.handle(session, handler);
        } else {
            mailHandler.handle(session, handler);
        }
    }
    // CSON: MethodLength


    public Cache<String, ConnInfo> connInfoCache() {
        return connInfoCache;
    }

    public static void main(final String... args)
        throws ConfigException, IOException
    {
        main(new ServerFactory(), args);
    }
}

