package ru.yandex.search.msal.contacts;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CodingErrorAction;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.http.HttpException;

import ru.yandex.dbfields.CollieFields;
import ru.yandex.http.server.sync.HttpSession;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.msal.AbstractReplicaHandler;
import ru.yandex.search.msal.pool.DBConnectionPool;

public class GetContactsHandler extends AbstractReplicaHandler {
    private static final String SELECT_STMT =
        "select contact_id, list_id, format, vcard, revision from "
            + " contacts.contacts where user_id = ? "
            + "and user_type = ?::contacts.user_type";
    private static final String WITH_CONTACT =
        SELECT_STMT + " and contact_id = ?";
    private static final String WITH_LIST_ID =
        SELECT_STMT + " and list_id = ?";
    private static final String SELECT_ALL =
        SELECT_STMT;

    public GetContactsHandler() {
        super(true);
    }

    // CSOFF: ParameterNumber
    // CSOFF: MagicNumber
    private void handleByContactId(
        final HttpSession session,
        final DBConnectionPool connPool,
        final String userType,
        final long uid,
        final long cid)
        throws HttpException, IOException, SQLException
    {
        try (Connection connection = connPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(WITH_CONTACT))
        {
            stmt.setLong(1, uid);
            stmt.setObject(2, userType);
            stmt.setLong(3, cid);

            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, session);
            }
        }
    }

    private void handleAll(
        final HttpSession session,
        final DBConnectionPool connPool,
        final String userType,
        final long uid)
        throws HttpException, IOException, SQLException
    {
        try (Connection connection = connPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(SELECT_ALL))
        {
            stmt.setLong(1, uid);
            stmt.setObject(2, userType);

            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, session);
            }
        }
    }

    private void handleByListId(
        final HttpSession session,
        final DBConnectionPool connPool,
        final String userType,
        final long uid,
        final long lid)
        throws HttpException, IOException, SQLException
    {
        try (Connection connection = connPool.getConnection(session.logger());
             PreparedStatement stmt = connection.prepareStatement(WITH_LIST_ID))
        {
            stmt.setLong(1, uid);
            stmt.setObject(2, userType);
            stmt.setLong(3, lid);

            try (ResultSet resultSet = stmt.executeQuery()) {
                writeResults(resultSet, session);
            }
        }
    }
    // CSON: MagicNumber
    // CSON: ParameterNumber

    @Override
    public void handlePgRequest(
        final HttpSession session,
        final DBConnectionPool connPool,
        final long uid)
        throws HttpException, IOException, SQLException
    {
        String userType = session.params().getString("userType");
        Long listId = session.params().getLong("listId", null);
        Long contactId = session.params().getLong("contactId", null);

        if (contactId != null) {
            handleByContactId(session, connPool, userType, uid, contactId);
        } else if (listId != null) {
            handleByListId(session, connPool, userType, uid, listId);
        } else {
            handleAll(session, connPool, userType, uid);
        }
    }

    protected void writeResults(
        final ResultSet resultSet,
        final HttpSession session)
        throws HttpException, IOException, SQLException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        JsonType type = JsonTypeExtractor.DOLLAR.extract(session.params());

        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     CharsetUtils.acceptedCharset(session.request())
                         .newEncoder()
                         .onMalformedInput(
                             CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(
                             CodingErrorAction.REPLACE));
             JsonWriter writer = type.create(outWriter))
        {
            writer.startArray();
            while (resultSet.next()) {
                writer.startObject();
                writer.key(CollieFields.CONTACT_ID);
                writer.value(resultSet.getLong(CollieFields.CONTACT_ID));
                writer.key(CollieFields.LIST_ID);
                writer.value(resultSet.getLong(CollieFields.LIST_ID));

                writer.key(CollieFields.VCARD);
                writer.value(resultSet.getString(CollieFields.VCARD));

                writer.key(CollieFields.FORMAT);
                writer.value(resultSet.getString(CollieFields.FORMAT));

                writer.key(CollieFields.REVISION);
                writer.value(resultSet.getLong(CollieFields.REVISION));

                writer.endObject();
            }
            writer.endArray();
        }

        session.response().setEntity(
            NByteArrayEntityFactory.INSTANCE.process(
                out.toByteArray()));
    }
}
