package ru.yandex.search.msal.processors;

import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;

import ru.yandex.search.msal.writers.ResultSetWriter;

public class TimestampFieldProcessor implements ResultSetFieldProcessor {
    private static final int BUFFER_SIZE = 32;
    private static final int TEN = 10;
    private static final long MILLIS = 1000L;

    private final char[] buf = new char[BUFFER_SIZE];
    private final String fieldName;

    public TimestampFieldProcessor(final String fieldName) {
        this.fieldName = fieldName;
    }

    @Override
    public void process(
        final ResultSet resultSet,
        final int columnIndex,
        final ResultSetWriter writer)
        throws IOException, SQLException
    {
        Timestamp timestamp = resultSet.getTimestamp(columnIndex);
        if (timestamp != null) {
            writer.write(fieldName, toString(buf, timestamp));
        }
    }

    private static String toString(
        final char[] buf,
        final Timestamp timestamp)
    {
        int nanos = timestamp.getNanos();
        int pos;
        if (nanos == 0) {
            pos = buf.length;
        } else {
            int fracLen = TEN - 1;
            while ((nanos % TEN) == 0) {
                --fracLen;
                nanos /= TEN;
            }
            for (int i = 1; i <= fracLen; ++i) {
                buf[buf.length - i] = (char) ((nanos % TEN) + '0');
                nanos /= TEN;
            }
            pos = buf.length - fracLen - 1;
            buf[pos] = '.';
        }
        long seconds = timestamp.getTime() / MILLIS;
        if (seconds == 0L) {
            buf[--pos] = '0';
        } else {
            while (seconds > 0L) {
                buf[--pos] = (char) ((seconds % TEN) + '0');
                seconds /= TEN;
            }
        }
        return new String(buf, pos, buf.length - pos);
    }

    public static String timestampToString(final Timestamp timestamp) {
        return toString(new char[BUFFER_SIZE], timestamp);
    }
}

