package ru.yandex.msearch.proxy;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpMessage;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.cache.async.AsyncCacheResultType;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.client.tvm2.UserAuthResult;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.http.server.async.LogRotateHandler;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.BaseServerConfigBuilder;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mail.search.subscriptions.SubscriptionsStatusHandler;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.broadcast.SelfBroadcastHandler;
import ru.yandex.msearch.proxy.api.async.enlarge.Your;
import ru.yandex.msearch.proxy.api.async.mail.Am22Handler;
import ru.yandex.msearch.proxy.api.async.mail.AttachmentsHandler;
import ru.yandex.msearch.proxy.api.async.mail.LinksHandler;
import ru.yandex.msearch.proxy.api.async.mail.MailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.chemodan.ChemodanInfoHandler;
import ru.yandex.msearch.proxy.api.async.mail.chemodan.ChemodanSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.classification.ClassificationHandler;
import ru.yandex.msearch.proxy.api.async.mail.dialogue.DialoguePersonalHandler;
import ru.yandex.msearch.proxy.api.async.mail.dialogue.DialogueThreadHandler;
import ru.yandex.msearch.proxy.api.async.mail.dialogue.DialoguesHandler;
import ru.yandex.msearch.proxy.api.async.mail.direct.DirectSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.furita.FuritaHandler;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.KeyboardHandler;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.KeyboardHandlerNoAuth;
import ru.yandex.msearch.proxy.api.async.mail.legacy.search.LegacyMailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.ml.MLSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.BasicMailSearchRelevanceFactory;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchRelevanceFactory;
import ru.yandex.msearch.proxy.api.async.mail.spaniel.SpanielSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.SubscriptionsCountHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.SubscriptionsHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.SubscriptionsListHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.SubscriptionsMailsHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.SubscriptionsUpdateStatusHandler;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.MigrationsTasksPostgresDao;
import ru.yandex.msearch.proxy.api.async.mail.tabs.count.FilterCountHandler;
import ru.yandex.msearch.proxy.api.async.mail.usertype.UserTypeChangeHandler;
import ru.yandex.msearch.proxy.api.async.senders.SendersHandler;
import ru.yandex.msearch.proxy.api.async.so.BugBountyGetScore;
import ru.yandex.msearch.proxy.api.async.so.GetDkimStats;
import ru.yandex.msearch.proxy.api.async.so.GetUserWeights;
import ru.yandex.msearch.proxy.api.async.so.ResolveStidHandler;
import ru.yandex.msearch.proxy.api.async.so.UpdateDkimStats;
import ru.yandex.msearch.proxy.api.async.so.UpdateWeights;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.folder.FolderSuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.history.HistorySuggestCleanupHandler;
import ru.yandex.msearch.proxy.api.async.suggest.history.HistorySuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.history.StoreHistorySuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.label.LabelSuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.subject.SubjectSuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.united.UnitedSuggestHandler;
import ru.yandex.msearch.proxy.api.async.suggest.zero.ZeroSuggestHandler;
import ru.yandex.msearch.proxy.api.chemodan.UserInfo;
import ru.yandex.msearch.proxy.config.ImmutableDkimStatsConfig;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.document.DiskDocumentHandler;
import ru.yandex.msearch.proxy.logger.ProxyTskvLogger;
import ru.yandex.msearch.proxy.socheck.SoCheck;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.mail.senders.SenderType;
import ru.yandex.passport.search.PassportMultidataSearchHandler;
import ru.yandex.search.msal.pool.DBConnectionPool;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.FilterPassiveStaterFactory;
import ru.yandex.stater.FunctionPassiveStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class AsyncHttpServer
    extends AsyncHttpServerBase<ImmutableMsearchProxyConfig>
{
    private static final Integer ZERO = 0;
    private static final Integer ONE = 1;

    private final ImmutableMsearchProxyConfig config;
    private final Synonyms synonyms;
    private final AsyncClient labelsClient;
    private final AsyncClient corpFilterSearchClient;
    private final AsyncClient corpLabelsClient;
    private final AsyncClient corpFoldersClient;
    private final AsyncClient soCheckClient;
    private final AsyncClient userSplitClient;
    private final AsyncClient iexProxyClient;
    private final AsyncClient corpMlClient;
    private final ErratumClient erratumClient;
    private final BlackboxClient blackboxClient;
    private final AsyncClient threadsClient;
    private final Tvm2TicketRenewalTask blackboxTvm2RenewalTask;
    private final BlackboxClient corpBlackboxClient;
    private final Tvm2TicketRenewalTask corpBlackboxTvm2RenewalTask;
    private final Tvm2TicketRenewalTask corpFilterSearchTvm2RenewalTask;
    private final AsyncClient proxyClient;
    private final SharedConnectingIOReactor fastReactor;
    private final AsyncClient fastSearchClient;
    private final RequestsStater backendsRequestsStater;
    private final RequestsStater filterSearchResponsesStater;
    private final RequestsStater searchBackendsResolveResponsesStater;
    private final RequestsStater searchFilterStater;
    private final TimeFrameQueue<Integer> searchBackendsResolvingsCountStats;
    private final List<TimeFrameQueue<Integer>> searchBackendsCountStats;
    private final TimeFrameQueue<Integer> queryLanguageHits;
    private final TimeFrameQueue<Integer> queryLanguageSimpleRequests;
    private final TimeFrameQueue<Integer> backendResponsesFreshness;
    private final TimeFrameQueue<Integer> malformedIndex;
    private final TimeFrameQueue<Integer> malformedIndexFresh;
    private final TimeFrameQueue<Integer> failedChemodanSid;
    private final TimeFrameQueue<SoCheckStat> soCheckStats;
    private final TimeFrameQueue<Long> producerStoreRequests;
    private final TimeFrameQueue<SenderType> sendersSenderType;
    private final TimeFrameQueue<SenderType> sendersSenderDomainType;
    private final TimeFrameQueue<SenderType> sendersPfilterType;
    private final TimeFrameQueue<SenderType> sendersTabpfType;
    private final TimeFrameQueue<AsyncCacheResultType> dkimStatsCacheHitType;
    private final UpstreamStater producerStoreStater;

    private final AsyncClient keyboardBackendClient;
    private final ProxyTskvLogger tskvLogger;
    private final MailSearchRelevanceFactory relevanceFactory;
    private final UpdateDkimStats updateDkimStatsHandler;

    private final MigrationsTasksPostgresDao migrationsTasksPostgresDao;

    public AsyncHttpServer(
        final ImmutableMsearchProxyConfig config,
        final Synonyms synonyms)
        throws ConfigException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(config);
        this.config = config;
        this.synonyms = synonyms;


        corpFilterSearchTvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("corpFilterSearchTvm2"),
            serviceContextRenewalTask,
            config.corpFilterSearchTvm2ClientConfig());

        //UserInfo.filterSearchTvm2RenewalTask(filterSearchTvm2RenewalTask);

        threadsClient =
            client("ThreadSearch", config.threadsConfig());

        labelsClient = client("LabelsClient", config.labelsConfig());
        corpFilterSearchClient =
            client("CorpFilterSearch", config.corpFilterSearchConfig());
        corpLabelsClient =
            client("CorpLabelsClient", config.corpLabelsConfig());
        corpFoldersClient =
            client("CorpFoldersClient", config.corpFoldersConfig());
        proxyClient = client("ProxyClient", config.proxyConfig());
        corpMlClient = client("MlClient", config.corpMlConfig());


        BaseServerConfigBuilder fastConfigBuilder =
            new BaseServerConfigBuilder(config);
        fastConfigBuilder.timerResolution(
                config.suggestConfig().searchClientResolution());
        ImmutableBaseServerConfig fastServerConfig =
            new ImmutableBaseServerConfig(
                fastConfigBuilder,
                config.dnsConfig(),
                config.tvm2ServiceConfig(),
                config.loggers(),
                config.staters(),
                config.auths(),
                config.limiters(),
                config.golovanPanel(),
                config.autoRegisterRequestStater());


        fastReactor = new SharedConnectingIOReactor(
            fastServerConfig,
            config.suggestConfig().dnsConfig(),
            new ThreadGroup(
                getThreadGroup(),
                config.name() + "-SuggestClient"));
        fastSearchClient =
            registerClient(
                "SuggestSearch",
                new AsyncClient(
                    fastReactor,
                    config.suggestConfig().searchClientConfig(),
                    RequestErrorType.ERROR_CLASSIFIER),
                config.suggestConfig().searchClientConfig());

        if (config.soCheckConfig() == null) {
            soCheckClient = null;
        } else {
            soCheckClient = client("SoCheck", config.soCheckConfig());
        }
        if (config.erratumConfig() == null) {
            erratumClient = null;
        } else {
            erratumClient =
                registerClient(
                    "Erratum",
                    new ErratumClient(reactor, config.erratumConfig()),
                    config.erratumConfig());
        }

        if (config.userSplitConfig() == null) {
            userSplitClient = null;
        } else {
            userSplitClient =
                client("UserSplitClient", config.userSplitConfig());
        }

        if (config.iexProxyConfig() == null) {
            iexProxyClient = null;
        } else {
            iexProxyClient =
                client("IexProxyClient", config.iexProxyConfig())
                    .adjustStater(
                        config.staters(),
                        new ru.yandex.http.util.request.RequestInfo(
                            new BasicHttpRequest(
                                RequestHandlerMapper.GET,
                                "/mail-search-snippet")));
        }

        blackboxClient =
            registerClient(
                "Blackbox",
                new BlackboxClient(reactor, config.blackboxConfig()),
                config.blackboxConfig());
        blackboxTvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("blackboxTvm2"),
            serviceContextRenewalTask,
            config.blackboxTvm2ClientConfig());

        UserInfo.blackboxTvm2RenewalTask(blackboxTvm2RenewalTask);

        corpBlackboxClient =
            registerClient(
                "CorpBlackbox",
                new BlackboxClient(reactor, config.corpBlackboxConfig()),
                config.corpBlackboxConfig());
        corpBlackboxTvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("corpBlackboxTvm2"),
            serviceContextRenewalTask,
            config.corpBlackboxTvm2ClientConfig());

        UserInfo.corpBlackboxTvm2RenewalTask(corpBlackboxTvm2RenewalTask);

        backendsRequestsStater =
            config.backendsResponsesStaterConfig().build();
        registerStater(backendsRequestsStater);

        filterSearchResponsesStater =
            config.filterSearchResponsesStaterConfig().build();
        registerStater(filterSearchResponsesStater);

        searchBackendsResolveResponsesStater =
            config.searchBackendsResolveResponsesStaterConfig().build();
        registerStater(searchBackendsResolveResponsesStater);

        searchFilterStater =
            config.searchFilterMailSearchStaterConfig().build();
        registerStater(searchFilterStater);

        searchBackendsResolvingsCountStats =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                searchBackendsResolvingsCountStats,
                new NamedStatsAggregatorFactory<>(
                    "backends-resolvings-count_ammm",
                    CountAggregatorFactory.INSTANCE)));
        malformedIndex = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                malformedIndex,
                new NamedStatsAggregatorFactory<>(
                    "malformed-index-count_ammm",
                    CountAggregatorFactory.INSTANCE)));

        malformedIndexFresh = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                malformedIndexFresh,
                new NamedStatsAggregatorFactory<>(
                    "malformed-index-fresh-count_ammm",
                    CountAggregatorFactory.INSTANCE)));

        searchBackendsCountStats =
            new ArrayList<>(config.maxExpectedSearchBackends() + 1);
        for (int i = 0; i <= config.maxExpectedSearchBackends(); ++i) {
            TimeFrameQueue<Integer> queue =
                new TimeFrameQueue<>(config.metricsTimeFrame());
            searchBackendsCountStats.add(queue);
            registerStater(
                new PassiveStaterAdapter<>(
                    queue,
                    new NamedStatsAggregatorFactory<>(
                        "backends-resolved-count-" + i + "_ammm",
                        CountAggregatorFactory.INSTANCE)));
        }

        backendResponsesFreshness =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                backendResponsesFreshness,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "backends-responses-fresh-count_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "backends-responses-fresh-total_ammm",
                        CountAggregatorFactory.INSTANCE))));

        queryLanguageHits = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                queryLanguageHits,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "query-language-hits_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "query-language-tries_ammm",
                        CountAggregatorFactory.INSTANCE))));

        queryLanguageSimpleRequests =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                queryLanguageSimpleRequests,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "query-language-simple-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "query-language-parsed-requests_ammm",
                        CountAggregatorFactory.INSTANCE))));

        producerStoreStater =
            new UpstreamStater(
                config.metricsTimeFrame(),
                "queue_pushes");
        registerStater(producerStoreStater);

        producerStoreRequests = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                producerStoreRequests,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "producer-store-requests_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "producer-store-failed_ammm",
                        CountAggregatorFactory.INSTANCE))));

        failedChemodanSid = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                failedChemodanSid,
                new NamedStatsAggregatorFactory<>(
                    "chemodan-sid-failed_ammm",
                    CountAggregatorFactory.INSTANCE)));

        soCheckStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                soCheckStats,
                new DuplexStaterFactory<>(
                    new DuplexStaterFactory<>(
                        new FilterPassiveStaterFactory<>(
                            x -> x.imap() == 0,
                            new DuplexStaterFactory<>(
                                new FilterPassiveStaterFactory<>(
                                    SoCheckStat::empty,
                                    new FunctionPassiveStaterFactory<>(
                                        SoCheckStat::result,
                                        new EnumStaterFactory<>(
                                            result ->
                                                "socheck-result-imap-0-empty-"
                                                + result + "_ammm",
                                            SoCheck.Result.values()))),
                                new FunctionPassiveStaterFactory<>(
                                    SoCheckStat::result,
                                    new EnumStaterFactory<>(
                                        result ->
                                            "socheck-result-imap-0-"
                                            + result + "_ammm",
                                        SoCheck.Result.values())))),
                        new FilterPassiveStaterFactory<>(
                            x -> x.imap() == 1,
                            new DuplexStaterFactory<>(
                                new FilterPassiveStaterFactory<>(
                                    SoCheckStat::empty,
                                    new FunctionPassiveStaterFactory<>(
                                        SoCheckStat::result,
                                        new EnumStaterFactory<>(
                                            result ->
                                                "socheck-result-imap-1-empty-"
                                                + result + "_ammm",
                                            SoCheck.Result.values()))),
                                new FunctionPassiveStaterFactory<>(
                                    SoCheckStat::result,
                                    new EnumStaterFactory<>(
                                        result ->
                                            "socheck-result-imap-1-"
                                            + result + "_ammm",
                                        SoCheck.Result.values()))))),
                    new FilterPassiveStaterFactory<>(
                        x -> x.imap() == 2,
                        new DuplexStaterFactory<>(
                            new FilterPassiveStaterFactory<>(
                                SoCheckStat::empty,
                                new FunctionPassiveStaterFactory<>(
                                    SoCheckStat::result,
                                    new EnumStaterFactory<>(
                                        result ->
                                            "socheck-result-imap-2-empty-"
                                            + result + "_ammm",
                                        SoCheck.Result.values()))),
                            new FunctionPassiveStaterFactory<>(
                                SoCheckStat::result,
                                new EnumStaterFactory<>(
                                    result ->
                                        "socheck-result-imap-2-"
                                        + result + "_ammm",
                                    SoCheck.Result.values())))))));

        sendersSenderType = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                sendersSenderType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    SenderType.values(),
                    "senders-sender-type-",
                    "senders",
                    "sender type",
                    null,
                    1)));

        sendersSenderDomainType =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                sendersSenderDomainType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    SenderType.values(),
                    "senders-sender-domain-type-",
                    "senders",
                    "sender domain type",
                    null,
                    1)));

        sendersPfilterType = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                sendersPfilterType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    SenderType.values(),
                    "senders-pfilter-type-",
                    "senders",
                    "pfilter sender type",
                    null,
                    1)));

        sendersTabpfType = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                sendersTabpfType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    SenderType.values(),
                    "senders-tabpf-type-",
                    "senders",
                    "Tab PF sender type",
                    null,
                    1)));

        if (config.tskvLogConfig() != null) {
            this.tskvLogger = new ProxyTskvLogger(config);
            registerLoggerForLogrotate(tskvLogger.logger());
        } else {
            this.tskvLogger = null;
        }

        relevanceFactory =
            new BasicMailSearchRelevanceFactory(this);

        keyboardBackendClient =
            client("KeyboardBackendClient", config.keyboardBackendConfig());

        if (config.pgPoolConfig() != null) {
            migrationsTasksPostgresDao = new MigrationsTasksPostgresDao(
                    new DBConnectionPool(
                            config.pgPoolConfig().url(),
                            config.pgPoolConfig(),
                            logger()
                    ),
                    logger()
            );
        } else {
            migrationsTasksPostgresDao = null;
        }

        register(new Pattern<>("/rotatelogs", false), new LogRotateHandler(this));
        register(
            new Pattern<>("/api/ora/delete/", false),
            new EmptyAsyncHandler(),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/api/am22/", false),
            new Am22Handler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/attachments", false),
            new AttachmentsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/links", false),
            new LinksHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/usertype", false),
            new UserTypeChangeHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/readtype", false),
            new SubscriptionsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/subscriptions", false),
            new SubscriptionsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/search", false),
            new MailSearchHandler(this, ""),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/ml/search", false),
            new MLSearchHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/spaniel/search", false),
            new SpanielSearchHandler(this),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/api/async/mail/search/direct", false),
            new DirectSearchHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/search/count", false),
            new FilterCountHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/classification", false),
            new ClassificationHandler(this),
            RequestHandlerMapper.GET, RequestHandlerMapper.POST);
        register(
            new Pattern<>("/api/async/senders", false),
            new SendersHandler(this),
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/api/async/suggest/history", false),
            new HistorySuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/contact", false),
            new ContactSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/history", false),
            new HistorySuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/history/cleanup", false),
            new HistorySuggestCleanupHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/history/save", false),
            new StoreHistorySuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/subject", false),
            new SubjectSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/label", false),
            new LabelSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/folder", false),
            new FolderSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest/zero", false),
            new ZeroSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/suggest", false),
            new UnitedSuggestHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/dialogues", false),
            new DialoguesHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/dialogue/personal", false),
            new DialoguePersonalHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/dialogue/thread", false),
            new DialogueThreadHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/keyboard", false),
            new KeyboardHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/keyboard-tvm", false),
            new KeyboardHandlerNoAuth(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/furita", false),
            new FuritaHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/enlarge/your", false),
            new Your(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/user/online", false),
//            new UserOnlineHandler(this),
            new Your(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/chemodan/search", false),
            new ChemodanSearchHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/chemodan_search", false),
            new ChemodanSearchHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/chemodan_info", false),
            new ChemodanInfoHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/broadcast/api", true),
            new SelfBroadcastHandler(this));

        register(
            new Pattern<>("/api/async/so/get-user-weights", false),
            new GetUserWeights(this));
        register(
            new Pattern<>("/api/async/so/update-user-weight", false),
            new UpdateWeights(this));
        register(
            new Pattern<>("/api/async/so/sobb-get-score", false),
            new BugBountyGetScore(this));

        // new subscriptions
        register(
            new Pattern<>("/api/async/mail/subscriptions/list", false),
            new SubscriptionsListHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/subscriptions/count", false),
            new SubscriptionsCountHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/subscriptions/update", false),
            new SubscriptionsUpdateStatusHandler(this),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("/api/async/mail/subscriptions/mails", false),
            new SubscriptionsMailsHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/api/async/mail/subscriptions/status", false),
            new SubscriptionsStatusHandler(this),
            RequestHandlerMapper.GET,
            RequestHandlerMapper.POST);
        if (config.multisearchConfig() != null) {
            this.register(
                new Pattern<>("/api/async/passport/multisearch", false),
                new PassportMultidataSearchHandler(this, config.multisearchConfig()));

            this.register(
                new Pattern<>("/api/async/passport/disk/document", false),
                new DiskDocumentHandler(this, config));
        }

        dkimStatsCacheHitType =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(
            new PassiveStaterAdapter<>(
                dkimStatsCacheHitType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    AsyncCacheResultType.values(),
                    "dkim-stats-cache-hit-type-",
                    "dkim",
                    "dkim cache hit type",
                    null,
                    1)));
        ImmutableDkimStatsConfig dkimStatsConfig = config.dkimStatsConfig();
        if (dkimStatsConfig == null || producerStoreClient == null) {
            updateDkimStatsHandler = null;
        } else {
            register(
                new Pattern<>("/api/async/so/get-dkim-stats", false),
                new GetDkimStats(this, dkimStatsConfig),
                RequestHandlerMapper.GET);

            updateDkimStatsHandler = new UpdateDkimStats(this);
            register(
                new Pattern<>("/api/async/so/update-dkim-stats", false),
                updateDkimStatsHandler,
                RequestHandlerMapper.GET);
        }
        register(
            new Pattern<>("/api/async/so/resolve-stid", false),
            new ResolveStidHandler(this),
            RequestHandlerMapper.GET);

        register(
            new Pattern<>("", true),
            new LegacyMailSearchHandler(this));
    }

    @Override
    public void start() throws IOException {
        fastReactor.start();
        blackboxTvm2RenewalTask.start();
        corpBlackboxTvm2RenewalTask.start();
        corpFilterSearchTvm2RenewalTask.start();
        if (updateDkimStatsHandler != null) {
            updateDkimStatsHandler.start();
        }
        super.start();
    }

    public TimeFrameQueue<Integer> failedChemodanSid() {
        return failedChemodanSid;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put("relevance", relevanceFactory.status(verbose));
        return status;
    }

    @Override
    public void close() throws IOException {
        try (UpdateDkimStats guarfd = updateDkimStatsHandler) {
            blackboxTvm2RenewalTask.cancel();
            corpBlackboxTvm2RenewalTask.cancel();
            corpFilterSearchTvm2RenewalTask.cancel();
        }
        super.close();
    }

    public AsyncClient furitaClient() {
        return furitaClient;
    }

    public AsyncClient tupitaClient() {
        return tupitaClient;
    }

    @Override
    public ImmutableMsearchProxyConfig config() {
        return config;
    }

    public Synonyms synonyms() {
        return synonyms;
    }

    public AsyncClient proxyClient() {
        return proxyClient;
    }

    public AsyncClient filterSearchClient(final boolean corp) {
        if (corp) {
            return corpFilterSearchClient;
        } else {
            return filterSearchClient;
        }
    }

    public AsyncClient labelsClient(final boolean corp) {
        if (corp) {
            return corpLabelsClient;
        } else {
            return labelsClient;
        }
    }

    public AsyncClient foldersClient(final boolean corp) {
        if (corp) {
            return corpFoldersClient;
        } else {
            return foldersClient;
        }
    }

    public AsyncClient mopsClient() {
        return mopsClient;
    }

    public AsyncClient soCheckClient() {
        return soCheckClient;
    }

    public AsyncClient userSplitClient() {
        return userSplitClient;
    }

    public AsyncClient iexProxyClient() {
        return iexProxyClient;
    }

    public ErratumClient erratumClient() {
        return erratumClient;
    }

    public BlackboxClient blackboxClient(
        final long uidOrSuid,
        final HttpContext context)
    {
        return blackboxClient(BlackboxUserinfo.corp(uidOrSuid), context);
    }

    public BlackboxClient blackboxClient(
        final boolean corp,
        final HttpContext context)
    {
        BlackboxClient client;
        if (corp) {
            client = corpBlackboxClient;
        } else {
            client = blackboxClient;
        }
        return client.adjust(context);
    }

    public String blackboxTvm2Ticket(final long uidOrSuid) {
        return blackboxTvm2Ticket(BlackboxUserinfo.corp(uidOrSuid));
    }

    public String blackboxTvm2Ticket(final boolean corp) {
        if (corp) {
            return corpBlackboxTvm2RenewalTask.ticket();
        } else {
            return blackboxTvm2RenewalTask.ticket();
        }
    }

    public String filterSearchTvm2Ticket(final boolean corp) {
        if (corp) {
            return corpFilterSearchTvm2RenewalTask.ticket();
        } else {
            return filterSearchTvm2RenewalTask.ticket();
        }
    }

    public UserAuthResult checkTvmUserAuthorization(HttpMessage request) {
        return serviceContextRenewalTask.checkUserAuthorization(
                request,
                YandexHeaders.X_YA_USER_TICKET);
    }

    public RequestsStater backendsRequestsStater() {
        return backendsRequestsStater;
    }

    public RequestsStater filterSearchResponsesStater() {
        return filterSearchResponsesStater;
    }

    public TimeFrameQueue<Integer> freshBackendsResponsesCountStats() {
        return backendResponsesFreshness;
    }

    public void resolvedSearchBackendsCount(
        final int count,
        final long startTime)
    {
        searchBackendsResolveResponsesStater.accept(
            new RequestInfo(
                TimeSource.INSTANCE.currentTimeMillis(),
                HttpStatus.SC_OK,
                startTime,
                startTime,
                0L,
                0L));
        searchBackendsResolvingsCountStats.accept(1);
        searchBackendsCountStats
            .get(Math.min(searchBackendsCountStats.size() - 1, count))
            .accept(1);
    }

    public void queryLanguageHit(final boolean successful) {
        queryLanguageHits.accept(successful ? ONE : ZERO);
    }

    public void queryLanguageSimpleRequest(final boolean simple) {
        queryLanguageSimpleRequests.accept(simple ? ONE : ZERO);
    }

    public void soCheckResult(
        final SoCheck.Result result,
        final boolean empty,
        final int imap)
    {
        soCheckStats.accept(new SoCheckStat(result, empty, imap));
    }

    public AsyncClient corpMlClient() {
        return corpMlClient;
    }

    public AsyncClient threadsClient() {
        return threadsClient;
    }

    public ProxyTskvLogger tskvLogger() {
        return tskvLogger;
    }

    public AsyncClient keyboardBackendClient() {
        return keyboardBackendClient;
    }

    public AsyncClient fastSearchClient() {
        return fastSearchClient;
    }

    public String resolveService(final String mdb, final Prefix prefix) {
        return resolveService(mdb, prefix, config);
    }

    public static String resolveService(
        final String mdb,
        final Prefix prefix,
        final ImmutableMsearchProxyConfig config)
    {
        if (prefix instanceof LongPrefix
            && BlackboxUserinfo.corp(((LongPrefix) prefix).prefix()))
        {
            if (ProxyParams.PG.equals(mdb)) {
                return config.pgCorpQueue();
            } else {
                return config.oracleCorpQueue();
            }
        } else {
            if (ProxyParams.PG.equals(mdb)) {
                return config.pgQueue();
            } else {
                return config.oracleQueue();
            }
        }
    }

    public void malformedIndex() {
        malformedIndex.accept(1);
    }

    public void malformedIndexFresh() {
        malformedIndexFresh.accept(1);
    }

    public MailSearchRelevanceFactory relevance() {
        return relevanceFactory;
    }

    public TimeFrameQueue<Long> producerStoreRequests() {
        return producerStoreRequests;
    }

    public UpstreamStater producerStoreStater() {
        return producerStoreStater;
    }

    public RequestsStater searchFilterStater() {
        return searchFilterStater;
    }

    public void senderType(final SenderType senderType) {
        sendersSenderType.accept(senderType);
    }

    public void senderDomainType(final SenderType senderType) {
        sendersSenderDomainType.accept(senderType);
    }

    public void pfilterSenderType(final SenderType senderType) {
        sendersPfilterType.accept(senderType);
    }

    public void tabPfSenderType(final SenderType senderType) {
        sendersTabpfType.accept(senderType);
    }

    public Consumer<AsyncCacheResultType> dkimStatsCacheHitType() {
        return dkimStatsCacheHitType;
    }

    public MigrationsTasksPostgresDao migrationsTasksDao() {
        return migrationsTasksPostgresDao;
    }

    private final class ChainedWarmupCallback
        implements FutureCallback<Object>
    {
        private final HttpHost nextHost;
        private final FutureCallback<Object> nextCallback;
        private final BasicAsyncRequestProducerGenerator generator;
        private final AtomicInteger counter;
        private final long deadline;

        public ChainedWarmupCallback(
            final HttpHost nextHost,
            final FutureCallback<Object> nextCallback,
            final BasicAsyncRequestProducerGenerator generator,
            final AtomicInteger counter,
            final long deadline)
        {
            this.nextHost = nextHost;
            this.nextCallback = nextCallback;
            this.generator = generator;
            this.deadline = deadline;
            this.counter = counter;
        }

        protected void next() {
            searchClient().execute(
                Collections.singletonList(nextHost),
                generator,
                deadline,
                EmptyAsyncConsumerFactory.ANY_GOOD,
                nextCallback);
        }

        @Override
        public void completed(final Object o) {
            counter.incrementAndGet();
            next();
        }

        @Override
        public void failed(final Exception e) {
            next();
        }

        @Override
        public void cancelled() {
            next();
        }
    }
}

