package ru.yandex.msearch.proxy;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.ByteBuffer;

import org.apache.http.nio.ContentEncoder;
import org.apache.http.nio.IOControl;

public class AsyncPrintStream extends PrintStream {
    private static final int BUFFER_SIZE = 8 * 1024;
    private final HttpServer.RequestContext ctx;
    private ByteBuffer writeBuffer = null;
    private ByteBuffer readBuffer = null;
    private long written = 0;
    private long read = 0;
    private IOControl ioctrl = null;
    private boolean closed = false;
    private boolean finished = false;
    //We can't synchronize on this because of
    //PrintStream is alread using syncrhonized methods
    private Object lock = new Object();

    private static final ByteArrayOutputStream fake =
        new ByteArrayOutputStream(1);

    public AsyncPrintStream(final HttpServer.RequestContext ctx) {
        super(fake);
        this.ctx = ctx;
    }

    public void write(int b) {
        //System.err.println("Write: " + b);
        synchronized(lock) {
            if (closed) {
                setError();
                return;
            }
            checkHeadersSent();
            if (ctx.isAborted()) {
                setError();
                lock.notify();
                return;
            }
            if (written - read >= BUFFER_SIZE) {
                flush(1);
            }
            if (writeBuffer.position() == writeBuffer.limit()) {
                writeBuffer.position(0);
                //System.err.println("writeb Flipped");
            }
            writeBuffer.put((byte)b);
            written++;
        }
        ioctrl.requestOutput();
    }

    public void write(byte buf[], int off, int len) {
//        System.err.println("len start: " + len);
        synchronized(lock) {
            if (closed) {
                setError();
                return;
            }
            checkHeadersSent();
            if (ctx.isAborted()) {
                setError();
                lock.notify();
                return;
            }
            while (len > 0) {
                int capacity = BUFFER_SIZE - (int)(written - read);
//                System.err.println("Capacity=" + capacity);
                if (capacity == 0) {
                    flush(Math.min(len, BUFFER_SIZE));
                }
                if (writeBuffer.position() == writeBuffer.limit()) {
                    writeBuffer.position(0);
//                    System.err.println("write Flipped");
                }
                int toCopy;
                if (writeBuffer.position() >= readBuffer.position()) {
                    toCopy =
                        Math.min(BUFFER_SIZE - writeBuffer.position(), len);
                    writeBuffer.put(buf, off, toCopy);
//                    System.err.println("WRITE1: toCopy=" + toCopy);
                } else {
                    toCopy =
                        Math.min(readBuffer.position() - writeBuffer.position(),
                            len);
                    writeBuffer.put(buf, off, toCopy);
//                    System.err.println("WRITE2: toCopy=" + toCopy);
                }
                written += toCopy;
                len -= toCopy;
                off += toCopy;
//                System.err.println("len iter: " + len);
                ioctrl.requestOutput();
            }
        }

    }

    private final void checkHeadersSent() {
//        System.err.println("checkHeadersSent start");
        if (writeBuffer == null) {
            writeBuffer = ByteBuffer.allocate(BUFFER_SIZE);
            readBuffer = writeBuffer.asReadOnlyBuffer();
        }
        if (ioctrl == null) {
            ctx.sendHeaders();
            ioctrl = ctx.getIOControl();
        }
//        System.err.println("checkHeadersSent end");
    }

    public void flush() {
        flush(BUFFER_SIZE);
    }

    public void flush(int count) {
        // count - bytes to be free after flush
        synchronized(lock) {
            if (closed) {
                return;
            }
            checkHeadersSent();
            count = Math.min(count, BUFFER_SIZE);
            while(!ctx.isAborted() &&
                BUFFER_SIZE - (written - read) < count)
            {
//                System.err.println("flush: count=" + count + ", w-r=" + (written-read));
                try {
                    ioctrl.requestOutput();
                    lock.wait();
                } catch (InterruptedException e) {
                }
            }
        }
    }

    public void writeContent(ContentEncoder encoder)
        throws IOException
    {
//        System.err.println("writeContent");
        synchronized(lock) {
            try {
                checkHeadersSent();
                if (finished) {
                    //System.err.println("writeContent.finish()");
                    encoder.complete();
                    return;
                }

                if (written - read > 0) {
//                    System.err.println("writeContent writtent-read=" + (written-read) + ", readPos=" + readBuffer.position() + ", writePos=" + writeBuffer.position());
                    if (writeBuffer.position() > readBuffer.position()) {
                        readBuffer.limit(writeBuffer.position());
                    } else {
                        readBuffer.limit(BUFFER_SIZE);
                    }
                    int encoded = encoder.write(readBuffer);
//                    System.err.println("encoded=" + encoded);
                    read += encoded;
                    if (readBuffer.position() == BUFFER_SIZE) {
                        readBuffer.position(0);
                    }
                    lock.notify();
                } else {
                    ioctrl.suspendOutput();
                }
                if (closed) {
                    lock.notify();
                }
            } catch (IOException e) {
                ctx.abort();
                lock.notify();
                throw e;
            }
        }
    }

    @Override
    public void close() {
        synchronized (lock) {
//            System.err.println("APS.close: closed=" + closed);
            lock.notify();
            if (closed) {
                return;
            }
            checkHeadersSent();
            flush();
            finished = true;
            if (ioctrl != null) {
                ioctrl.requestOutput();
            }
            closed = true;
        }
    }

}
