package ru.yandex.msearch.proxy;

import java.io.IOException;
import java.io.PrintStream;
import java.io.StringWriter;

import java.util.Map;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.LinkedHashSet;
import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.http.util.YandexHttpStatus;

import ru.yandex.json.writer.HumanReadableJsonWriter;

import ru.yandex.msearch.proxy.api.mail.rules.StoreSearchRequestAdapter;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.collector.SortingCollector;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.searchmap.SearchMap;
import ru.yandex.msearch.proxy.api.Api;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.search.Searcher;
import ru.yandex.msearch.proxy.search.ServerFactory;
import ru.yandex.msearch.proxy.HttpServer.HttpException;
import ru.yandex.msearch.proxy.socheck.FakeSoCheck;
import ru.yandex.msearch.proxy.socheck.SoCheck;
import ru.yandex.msearch.proxy.socheck.SoCheckFactory;

public class Handler implements HttpHandler {
    public static final int USERS_PER_SEARCH = 1000;
    private static final ConcurrentHashMap<String, AtomicInteger> rateLimitMap =
        new ConcurrentHashMap<String, AtomicInteger>(100,
            (float) 0.5,
            Runtime.getRuntime().availableProcessors() * 2);

    private final SearchMap searchMap = SearchMap.getInstance();
    private final Api api;
    private final SoCheckFactory soCheckFactory;
    private final ImmutableMsearchProxyConfig config;
    private final int maxThreads;
    private final AtomicInteger totalRequestsInFlight = new AtomicInteger(0);
    private final AtomicLong botNetRequestNumber = new AtomicLong();
    private final Pattern forbiddenRequests;
    private final StoreSearchRequestAdapter storeRequestAdapter;

    public Handler(
        final Api api,
        final SoCheckFactory soCheckFactory,
        final ImmutableMsearchProxyConfig config)
    {
        this.api = api;
        this.soCheckFactory = soCheckFactory;
        this.config = config;
        this.maxThreads = config.syncServerThreads();
        this.forbiddenRequests = config.forbiddenRequests();
        if (config.indexSearchRequests()
            && config.producerClientConfig() != null
            && soCheckFactory != null)
        {
            storeRequestAdapter = StoreSearchRequestAdapter.create(
                soCheckFactory.reactor(),
                config);
        } else {
            storeRequestAdapter = null;
        }
    }

    public int handleRequest(
        HttpServer.RequestContext ctx,
        String request,
        String requestType,
        HttpServer.HttpParams params,
        HttpServer.HttpHeaders headers,
        PrintStream ps) throws MsearchProxyException
    {
        ctx.log.warn("SyncHandler " + request);
        request = request.replaceFirst("^[\\/]+", "/");
        int requestEnd = request.indexOf('?');
        if (requestEnd == -1) {
            requestEnd = request.length();
        }

        if (request.equals("/statusold")) {
            int total = totalRequestsInFlight.get();
            for (Map.Entry<String, AtomicInteger> entry : rateLimitMap
                .entrySet()) {
                String req = entry.getKey();
                AtomicInteger perCount = entry.getValue();
                int max = (int) ((float) maxThreads * 0.7);
                if (total >= (int) ((float) maxThreads * 0.95)) {
                    max = (int) ((float) maxThreads * 0.05);
                }
                ps.println(req + ": " + perCount + "/" + max);
            }
            ctx.log.info("/status request");
            return 0;
        }

        if (request.equals("/cacheinfo")) {
            ps.println(api.suggest().cacheSize() + " entries in cache");
            ps.println(api.suggest().cachekeySet());
            ctx.log.info("/cacheinfo request");
            return 0;
        }

        if (request.equals("/socheckstatus") && soCheckFactory != null) {
            StringWriter sw = new StringWriter();
            try {
                new HumanReadableJsonWriter(sw).value(soCheckFactory.status());
            } catch (IOException e) {
                throw new MsearchProxyException("Error in /socheckstatus", e);
            }
            ps.println(sw.toString());
            ctx.log.info("/socheckstatus request");
            return 0;
        }

        String rateLimitKey = request.substring(0, requestEnd);
        AtomicInteger perRequestCount = rateLimitMap.get(rateLimitKey);
        if (perRequestCount == null) {
            AtomicInteger newCount = new AtomicInteger(0);
            perRequestCount = rateLimitMap.putIfAbsent(rateLimitKey, newCount);
            if (perRequestCount == null) {
                perRequestCount = newCount;
            }
        }

        try {
            int perCount = perRequestCount.incrementAndGet();
            if (perCount > (int) ((float) maxThreads * 0.7) ||
                (totalRequestsInFlight.incrementAndGet() >= (int) (
                    (float) maxThreads * 0.95) && perCount > (int) (
                    (float) maxThreads * 0.05))
                )
            {
                ctx.log.err("request: " + request
                    + " rejected because of rate limit. Total requests of "
                    + "this type: "
                    + perCount);
                ctx.setHttpCode(YandexHttpStatus.SC_TOO_MANY_REQUESTS);
                ps.println("request: " + request
                    + " rejected because of rate limit. Total requests of "
                    + "this type: "
                    + perCount);
                return 0;
            }

            if (request.startsWith("/api/")
                || request.startsWith("/apinoxml/"))
            {
                try {
                    return api.dispatch(
                        ctx,
                        request,
                        requestType,
                        params,
                        headers,
                        ps);
                } catch (ApiException e) {
                    ctx.setHttpCode(e.statusCode());
                    ctx.log.err(Logger.exception(e));
                    ps.println("Code 2319");
//	            ps.println(Logger.exception(e));
                    return 0;
                }
            }

            if (request.equals("/pingold")) {
                ctx.log.info("ping-pong");
                ps.println("pong");
                return 0;
            }

            SoCheck soCheck;
            if (soCheckFactory == null) {
                soCheck = FakeSoCheck.INSTANCE;
            } else {
                soCheck = soCheckFactory.create(params, ctx);
            }
            return handleProxySearch(ctx, params, ps, soCheck);
        } finally {
            perRequestCount.decrementAndGet();
            totalRequestsInFlight.decrementAndGet();
        }
    }

    private int handleProxySearch(
        HttpServer.RequestContext ctx,
        HttpServer.HttpParams params,
        PrintStream ps,
        SoCheck soCheck) throws MsearchProxyException
    {
        String db = params.get("db");
        String user = params.get("user");
        String fids = params.get("fids");

        if (db == null) {
            ps.println("db parameter is empty");
            ctx.log.err("db parameter is empty");
            return 500;
        }

        if (user == null) {
            ps.println("user parameter is empty");
            ctx.log.err("empty request");
            return 500;
        }

        // raw request for storing
        final String rawRequest = params.get("text");

        if (params.get("scope") != null) {
            String scope = params.get("scope");
            if (scope.equals("history")) {
                params.replace("imap", "1");
                String text = params.get("text");
                if (text != null) {
                    String[] emails = text.split(" ");
                    text = "";
                    for (int i = 0; i < emails.length; i++) {
                        if (i > 0) {
                            text += " OR ";
                        }
                        text += "hdr_from:\"" + emails[i] + "\" OR hdr_to:\""
                            + emails[i] + "\"";
                    }
                    params.replace("text", text);
                }
            } else if (scope.equals("rpopid")) {
                params.replace("imap", "1");
                String text = params.get("text");
                if (text != null) {
                    text = text.trim();
                    params.replace(
                        "text",
                        "headers:\"_$XHIDENX$_x-yandex-rpop-id: " + text
                            + "\" OR headers:\"x-yandex-rpop-id: " + text
                            + '"');
                    params.remove("scope");
                }
            } else if (scope.equals("rpopinfo")) {
                params.replace("imap", "1");
                String text = params.get("text");
                if (text != null) {
                    text = text.trim();
                    params.replace(
                        "text",
                        "headers:\"_$XHIDENX$_x-yandex-rpop-info: " + text
                            + "\" OR headers:\"x-yandex-rpop-info: " + text
                            + '"');
                    params.remove("scope");
                }
            } else if (scope.equals("imap")) {
                params.replace("imap", "1");
                params.remove("scope");
                String text = params.get("text");
                if (text != null) {
                    params.replace(
                        "text",
                        text.replace("*", "").replace("?", ""));
                }
            } else if (scope.equals("body_text")) {
                params.replace("scope", "pure_body");
            }
        }

        String imap = params.get("imap");
        if (imap != null && imap.equals("1")) {
            String text = params.get("text");
            if (text != null && forbiddenRequests != null) {
                if (forbiddenRequests.matcher(text).matches()) {
                    ctx.log.err("banned request");
                    ctx.setHttpCode(403);
                    ps.println("request '" + text + "' is forbidden by "
                        + forbiddenRequests);
                    return 0;
                }
            }
        }

        String text = params.get("text");
        if (text == null || text.trim().isEmpty()) {
            if ("pg".equals(db)) {
                text = "uid:" + user;
            } else {
                text = "suid:" + user;
            }
            params.replace("text", text);
            params.replace("imap", "1");
        }

        Collector collector = new SortingCollector("mid", "received_date");
        Searcher searcher =
            new Searcher(ctx, USERS_PER_SEARCH, searchMap, params, collector);

        String[] filterFids = null;
        List<String> fidsList = new LinkedList<String>();
        if (fids != null && fids.trim().length() > 0) {
            filterFids = fids.split(",");
            fidsList.addAll(Arrays.asList(filterFids));
        }
        String scope = params.get("search_scope");
        boolean includeSelf = true;
        boolean corp = BlackboxUserinfo.corp(Long.parseLong(user));
        if (corp && !fidsList.isEmpty()) {
            includeSelf = false;
        }
        String[] users = new String[] {user};
        if (corp) {
            db = config.pgCorpQueue();
        } else {
            db = config.pgQueue();
        }
        searcher.setAccountCompleted(true);
        searcher.search(db, users);
        if (!searcher.isCompleted(users)) {
            ctx.log.err("search is not completed for some users");
            ctx.setHttpCode(500);
            ps.println("search in not completed for some users");
            return 500;
        }
        int found = collector.getTotalCount();
        ctx.log.info("found " + collector.getTotalCount() + " documents");
        SoCheck.Result soCheckResult = soCheck.result(found);
        ctx.log.info("socheck: check result is: " + soCheckResult);
        if (soCheckResult == SoCheck.Result.SPAM) {
            if (soCheckFactory.config().banSpam()) {
                ctx.log.info("Looks like bot-net, dropping search results");
                String fakeRequest = soCheckFactory.config().fakeRequest();
                long fakeRequestInterval =
                    soCheckFactory.config().fakeRequestInterval();
                long botNetRequestNumber =
                    this.botNetRequestNumber.incrementAndGet();
                if (fakeRequest != null
                    && !fakeRequest.isEmpty()
                    && fakeRequestInterval > 0L
                    && botNetRequestNumber % fakeRequestInterval == 0L)
                {
                    ctx.log.info("Performing fake request");
                    params.replace("text", fakeRequest);
                    return handleProxySearch(
                        ctx,
                        params,
                        ps,
                        FakeSoCheck.INSTANCE);
                } else {
                    ctx.log.info("Returning empty SERP");
                    collector = new SortingCollector("mid", "received_date");
                }
            } else {
                ctx.log.info("ban_spam is not set, ignoring SO check result");
            }
        }
        if (storeRequestAdapter != null) {
            storeRequestAdapter.storeRequest(ctx, params, "slash_question",
                rawRequest, found == 0);
        }
        OutputPrinter printer =
            new OutputPrinter(ctx.getSessionId(), params, ps);
        int count = printer.print(collector);
        return count;
    }
}
