package ru.yandex.msearch.proxy;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.ProtocolVersion;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.collection.Pattern;

import ru.yandex.http.server.async.BaseAsyncServer;

import ru.yandex.http.util.YandexReasonPhraseCatalog;

import ru.yandex.http.util.nio.BasicAsyncResponseProducer;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.http.util.server.LoggingServerConnection;

import ru.yandex.msearch.proxy.api.Api;
import ru.yandex.msearch.proxy.api.chemodan.UserInfo;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.socheck.SoCheckFactory;

import ru.yandex.parser.string.BooleanParser;

public class HttpServer extends AsyncHttpServer {
    public static final int MAX_IDLE_TIME = 1000 * 120;
    private static final ProtocolVersion HTTP11 =
        new ProtocolVersion("HTTP", 1, 1);

    private final SoCheckFactory soCheckFactory;
    private final HttpWorkerPool pool;

    public HttpServer(final ImmutableMsearchProxyConfig config, final Api api)
        throws Exception
    {
        super(config, api.synonyms());

        logger().info(
            "Starting HttpWorkerPool: threads=" + config.syncServerThreads()
            + ", queueSize=" + config.syncServerQueueSize());

        if (config.soCheckConfig() == null) {
            soCheckFactory = null;
        } else {
            soCheckFactory =
                new SoCheckFactory(reactor, config.soCheckConfig());
        }
        Handler handler = new Handler(api, soCheckFactory, config);

        pool = new HttpWorkerPool(
            handler,
            config.syncServerThreads(),
            config.syncServerQueueSize());

        register(new Pattern<>("/api", true), new SyncCodeHandler(pool, this));
        register(
            new Pattern<>("/statusold", true),
            new SyncCodeHandler(pool, this),
            false);
        register(
            new Pattern<>("/cacheinfo", false),
            new SyncCodeHandler(pool, this),
            false);
        register(
            new Pattern<>("/socheckstatus", false),
            new SyncCodeHandler(pool, this),
            false);
        register(
            new Pattern<>("/pingold", false),
            new SyncCodeHandler(pool, this),
            false);
    }

    @Override
    public void start() throws IOException {
        if (soCheckFactory != null) {
            soCheckFactory.start();
        }
        pool.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        try (Closeable soCheckFactory = this.soCheckFactory;
            Closeable pool = this.pool)
        {
            super.close();
        }
    }

    public static class RequestContext
    {
        public final AsyncPrintStream ps;
        public final  PrefixingLogger log;
        private final String sessionId;
        private int httpCode;
        private String contentType;
        private byte[] postData;
        private IOControl ioctrl = null;
        private final AtomicBoolean headersSent = new AtomicBoolean();
        private BasicAsyncResponseProducer responseProducer = null;
        private volatile boolean isAborted = false;

        private final HttpWorkerPool.HttpWork httpWork;

        public RequestContext(final HttpWorkerPool.HttpWork httpWork) {
            this.httpWork = httpWork;

            ps = new AsyncPrintStream(this);
            log =
                new PrefixingLogger(
                    httpWork.context.getAttribute(
                        BaseAsyncServer.LOGGER,
                        Logger.class));
            sessionId = httpWork.context.getAttribute(
                BaseAsyncServer.SESSION_ID,
                String.class);
            httpCode = HttpStatus.SC_OK;
            contentType = "text/plain";
        }

        public AsyncHttpServer server() {
            return httpWork.server;
        }

        public LoggingServerConnection conn() {
            return (LoggingServerConnection) httpWork.context.getConnection();
        }

        public long requestStartTime() {
            return conn().requestStartTime();
        }

        private String connectionHdr() {
            Header hdr = httpWork.request.getFirstHeader("Connection");
            if (hdr != null && hdr.getValue() != null) {
                return hdr.getValue();
            }
            ProtocolVersion requestVersion =
                httpWork.request.getProtocolVersion();
            if (requestVersion.greaterEquals(HttpServer.HTTP11)) {
                return "Keep-Alive";
            } else {
                return "Close";
            }
        }

        private boolean canChunked() {
            ProtocolVersion requestVersion =
                httpWork.request.getProtocolVersion();
            if (requestVersion.greaterEquals(HttpServer.HTTP11)) {
                return connectionHdr().equalsIgnoreCase("Keep-Alive");
            }
            return false;
        }

        public void sendHeaders() {
            if (!headersSent.get()) {
                SyncCodeHttpEntity entity =
                    new SyncCodeHttpEntity(this, canChunked());
                String reason =
                    YandexReasonPhraseCatalog.INSTANCE.getReason(
                        httpCode, Locale.US);
                HttpResponse response =
                    new BasicHttpResponse(HttpVersion.HTTP_1_1,
                        httpCode,
                        reason);
                response.setEntity(entity);
                entity.setContentType(
                    new BasicHeader("Content-Type", contentType));

                responseProducer = new BasicAsyncResponseProducer(response) {
                    @Override
                    public void failed(Exception e) {
//                        System.err.println("FAILED: " + e);
                        abort();
                    }
                };
                if (headersSent.compareAndSet(false, true)) {
                    httpWork.exchange.submitResponse(responseProducer);
                    httpWork.exchange.setTimeout(0);
                } else {
                    log.err("Suppressing response: " + responseProducer);
                    try {
                        responseProducer.close();
                    } catch (IOException e) {
                        log.warn("Failed to consume response producer", e);
                    }
                }
            }
            waitIOControl();
        }

	public String getSessionId()
	{
	    return sessionId;
	}

	public void setHttpCode( int code )
	{
	    httpCode = code;
	}

	public int getHttpCode()
	{
	    return httpCode;
	}

        public void checkAbort() {
        }

        public boolean isAborted() {
            return isAborted;
        }

        public void abort() {
            isAborted = true;
            if (ps != null) {
                ps.close();
            }
            synchronized(this) {
                notify();
            }
        }

        public synchronized void setIOControl(IOControl ioctrl) {
            this.ioctrl = ioctrl;
            notify();
        }

        public IOControl getIOControl() {
            return ioctrl;
        }

        public synchronized void waitIOControl() {
            while(ioctrl == null && !isAborted) {
                try {
                    wait(100);
                } catch (InterruptedException e) {

                }
            }
        }

	private void doSetContentType( String type )
	{
	    contentType = type;
	}

	public void setContentType( Header header )
	{
	    doSetContentType( header.getValue() );
	}

	public void setContentType( String type )
	{
	    doSetContentType( type + "; charset=utf-8" );
	}

	public final void setPostData(byte[] data) {
	    postData = data;
	}

	public final byte[] getPostData() {
	    return postData;
	}

        public void close(Throwable t) throws IOException {
            if (headersSent.compareAndSet(false, true)) {
                String msg = ru.yandex.msearch.proxy.logger.Logger.exception(t);
                httpWork.exchange.getResponse().setEntity(
                    new NStringEntity(msg, ContentType.TEXT_PLAIN));
                httpWork.exchange.submitResponse();
            } else {
                responseProducer.close();
            }
        }

        public PrintStream ps() {
            return ps;
        }

        public Header referer() {
            return AsyncClient.extractReferer(httpWork.request);
        }
    }

    public static class PrefixingLogger
    {
        private final Logger logger;

        public PrefixingLogger(final Logger logger) {
            this.logger = logger;
        }

        public void debug(final String msg) {
            logger.fine(msg);
        }

        public void debug(final String msg, final Throwable thrown) {
            logger.log(Level.FINE, msg, thrown);
        }

        public void info(final String msg) {
            logger.info(msg);
        }

        public void info(final String msg, final Throwable thrown) {
            logger.log(Level.INFO, msg, thrown);
        }

        public void warn(final String msg) {
            logger.warning(msg);
        }

        public void warn(final String msg, final Throwable thrown) {
            logger.log(Level.WARNING, msg, thrown);
        }

        public void err(final String msg) {
            logger.severe(msg);
        }

        public void err(final String msg, final Throwable thrown) {
            logger.log(Level.SEVERE, msg, thrown);
        }
    }

    public static class HttpHeaders extends HashMap<String,String>
    {
    }

    public static class HttpParams
    {
    private Map<String, List<String>> params;
	public HttpParams()
	{
	    this( new LinkedHashMap<String, List<String>>() );
	}

        public HttpParams( Map<String, List<String>> params )
        {
            this.params = params;
        }

        public HttpParams copy()
        {
            Map<String, List<String>> params = new LinkedHashMap<>();
            for( Map.Entry<String, List<String>> entry: this.params.entrySet() )
            {
                params.put( entry.getKey(), new ArrayList<>(entry.getValue()) );
            }
            return new HttpParams( params );
        }

	public void clear()
	{
	    params.clear();
	}

	public Iterator<String> keysIterator()
	{
	    return params.keySet().iterator();
	}

        public Iterator<Map.Entry<String, List<String>>> iterator() {
            return params.entrySet().iterator();
        }

        public Set<Map.Entry<String, List<String>>> entrySet() {
            return params.entrySet();
        }

        public void addAll(Map<String,String> params) {
            for (Map.Entry<String,String> entry : params.entrySet()) {
                add(entry.getKey(), entry.getValue());
            }
        }

	public void add( String key, String value )
	{
            if( value == null ) return;
	    List<String> list = params.get( key );
	    if( list == null )
	    {
		list = new ArrayList<String>();
		params.put( key, list );
	    }
	    list.add( value );
	}

	public int count( String key )
	{
	    List<String> list = params.get( key );
	    if( list != null ) return list.size();
	    return 0;
	}

	public String get( String key, int index )
	{
	    List<String> list = params.get( key );
	    if( list == null ) return null;
	    if( list.size() <= index ) return null;
	    return list.get(index);
	}

	public String get( String key )
	{
	    List<String> list = params.get( key );
	    if( list == null || list.isEmpty() ) return null;
	    return list.get( list.size() - 1 );
	}

	public String getSafe( String key )
	{
	    String value = get( key );
	    if( value == null ) return "";
	    return value;
	}

        public Integer getInt(String key) {
            String value = get(key);
            if (value == null) return null;
            return Integer.parseInt(value);
        }

        public int getInt(String key, int defaultValue) {
            String value = get(key);
            if (value == null) return defaultValue;
            return Integer.parseInt(value);
        }

        public boolean getBoolean(String key, boolean defaultValue) {
            String value = get(key);
            if (value == null) return defaultValue;
            try {
                return BooleanParser.INSTANCE.apply(value);
            } catch (Exception e) {
                return defaultValue;
            }
        }

	public List<String> getAll( String key )
	{
	    return params.get(key);
	}

	public List<String> getAllOrEmpty( String key )
	{
	    List<String> all = params.get(key);
            if (all == null) {
                all = Collections.emptyList();
            }
            return all;
	}

	public void replace( String key, String value )
	{
	    List<String> list = params.get( key );
	    if( list == null )
	    {
		list = new ArrayList<String>();
		params.put( key, list );
	    }
	    else
	    {
		list.clear();
	    }
            if( value != null ) list.add( value );
	}

	public void remove( String key )
	{
	    params.remove( key );
	}

        @Override
        public String toString()
        {
            return params.toString();
        }
    }

    public static final class HttpException extends IOException
    {
	public HttpException( String msg )
	{
	    super( msg );
	}
    }

}
