package ru.yandex.msearch.proxy;

/* An example of a very simple, multi-threaded HTTP server.
 * Implementation notes are in WebServer.html, and also
 * as comments in the source code.
 */

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.net.URLDecoder;

import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.util.EntityUtils;

import ru.yandex.msearch.proxy.logger.Logger;

class HttpWorker implements Runnable {
    private HttpHandler handler;
    private HttpWorkerPool server;
    private volatile boolean exit = false;

    private String requestType;
    private String requestString;
    private String requestVersion;
    private HttpServer.HttpHeaders headers;
    private HttpServer.HttpParams httpParams;
    private AsyncPrintStream ps;
    private HttpServer.RequestContext ctx;
    private HttpWorkerPool.HttpWork work = null;

    HttpWorker(HttpWorkerPool server, HttpHandler handler) {
	this.handler = handler;
	this.server = server;
	this.headers = new HttpServer.HttpHeaders();
	this.httpParams = new HttpServer.HttpParams();
    }

    private synchronized boolean waitForTask() {
        if (exit) return false;
        if (work != null) return true;
        while(true) {
            long idleStart = System.currentTimeMillis();
            while (work == null && !exit &&
                (System.currentTimeMillis() - idleStart) < HttpServer.MAX_IDLE_TIME)
            {
                try {
                    long waitTime = HttpServer.MAX_IDLE_TIME -
                        (System.currentTimeMillis() - idleStart);
                    wait(waitTime);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                    work = null;
                    return false;
                }
            }
            if (exit) {
                return false;
            }
            if (work == null) {
                boolean expired = server.tryExpireWorker(this);
                if (expired) {
                    return false;
                } else {
                    continue;
                }
            }
            return true;
        }
    }

    public void run() 
    {
        while(!exit)
        {
            if (!waitForTask()) {
                return;
            }
            try {
                handleClient();
            } catch( Throwable e ) {
                e.printStackTrace();
                if (ctx != null) {
                    ctx.log.err("HttpWorker failed to handle client: " + e.toString());
            	    ctx.log.err("HttpWorker stackTrace: " + Logger.exception(e));
            	    try {
            	        ctx.close(e);
            	    } catch (IOException ign) {
            	    }
        	} else {
            	    Logger.err("HttpWorker failed to handle client: " + e.toString());
            	    Logger.err("HttpWorker stackTrace: " + Logger.exception(e));
            	}
            } finally {
                work = null;
                ctx = null;
                server.freeWorker(this);
            }
        }
    }

    public synchronized void setWork(HttpWorkerPool.HttpWork work) {
        this.work = work;
        notify();
    }

    public synchronized void stop()
    {
	exit = true;
	notify();
    }

    public boolean isAborted() {
        return ctx.isAborted();
    }

    String decodeString(String str) {
        try {
            return URLDecoder.decode(str, "utf8");
        } catch (java.io.UnsupportedEncodingException e) {
            return str;
        }
    }

    void parseHttpParams(String line) {
        int q = line.indexOf('?');
        if (q != -1) {
            line = line.substring(q + 1);
        }
	String[] params = line.split("&");
	for (int i = 0; i < params.length; i++) {
	    int sep = params[i].indexOf('=');
	    if (sep == -1) {
		httpParams.add(params[i], "");
	    } else {
		httpParams.add(params[i].substring(0, sep),
		    decodeString(params[i].substring(sep + 1)));
	    }
	}
    }

    void clearHeaders() {
	headers.clear();
    }

    void clearHttpParams() {
	httpParams.clear();
    }

    void handleClient() throws IOException {
        ctx = new HttpServer.RequestContext(work);
        this.ps = ctx.ps;

        int retval = 0;
	try
	{
	    requestString = work.request.getRequestLine().getUri();
	    requestType = work.request.getRequestLine().getMethod();
	    parseHttpParams(requestString);
	    HeaderIterator hiter = work.request.headerIterator();
	    while (hiter.hasNext()) {
	        Header header = hiter.nextHeader();
	        headers.put(header.getName().toLowerCase(), header.getValue());
	    }
	    if (work.request instanceof HttpEntityEnclosingRequest) {
	        HttpEntity entity =
	            ((HttpEntityEnclosingRequest)work.request).getEntity();
	        if (entity != null) {
	            ctx.setPostData(EntityUtils.toByteArray(entity));
	            String contentType = entity.getContentType().getValue();
	            if (contentType.startsWith(
	                "application/x-www-form-urlencoded"))
	            {
	                BufferedReader reader =
	                    new BufferedReader(
	                        new InputStreamReader(
	                            new ByteArrayInputStream(
	                                ctx.getPostData()), "UTF-8"));
	                String line;
	                while((line = reader.readLine()) != null) {
	                    parseHttpParams(line);
	                }
	            }
	        }
	    }
            retval = handler.handleRequest(
                ctx,
                requestString,
                requestType,
                httpParams,
                headers, ps);
        } catch(Exception e) {
            ctx.setHttpCode(500);
            ps.println("failed to handle request: " + Logger.exception(e));
            ctx.log.err("failed to handle request: " + e.toString());
            ctx.log.err(Logger.exception(e));
        } finally {
            final long time =
                System.currentTimeMillis() - ctx.requestStartTime();
            ctx.log.info("execution time: " + requestString +
                " : " + time + " : " + retval);
            ctx.conn().setHitsCount(Integer.toString(retval));
            ps.close();
            ps = null;
            headers.clear();
            httpParams.clear();
    	    requestString = null;
        }
    }
}
