package ru.yandex.msearch.proxy;

/* An example of a very simple, multi-threaded HTTP server.
 * Implementation notes are in WebServer.html, and also
 * as comments in the source code.
 */

import java.io.Closeable;

import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

public class HttpWorkerPool implements Closeable {
    public static final int MAX_IDLE_TIME = 1000 * 120;

    private final ConcurrentHashMap<HttpWorker,HttpWorker> activeWorkers =
        new ConcurrentHashMap<HttpWorker,HttpWorker>();
    private final LinkedBlockingDeque<HttpWorker> freeWorkers =
        new LinkedBlockingDeque<HttpWorker>();

    private final AtomicInteger id = new AtomicInteger(0);

    private final ArrayBlockingQueue<HttpWork> workQueue;
    private volatile boolean exit = false;
    private final HttpHandler handler;
    private final Thread queueDrainer;

    private final int minWorkers;

    public static class HttpWork {
        public final HttpRequest request;
        public final HttpAsyncExchange exchange;
        public final HttpCoreContext context;
        public final AsyncHttpServer server;
        public HttpWork(
            final HttpRequest request,
            final HttpAsyncExchange exchange,
            final HttpContext context,
            final AsyncHttpServer server)
        {
            this.request = request;
            this.exchange = exchange;
            this.context = HttpCoreContext.adapt(context);
            this.server = server;
        }

        public void fail(String msg, int code) {
            exchange.getResponse().setEntity(
                new NStringEntity(msg, ContentType.TEXT_PLAIN));
            exchange.getResponse().setStatusCode(code);
            exchange.submitResponse();
        }
    }

    public HttpWorkerPool(
        final HttpHandler handler,
        final int workers,
        final int queueSize)
    {
        this.handler = handler;
	minWorkers =
	    Math.max(workers / 10, Runtime.getRuntime().availableProcessors());
        workQueue = new ArrayBlockingQueue<HttpWork>(queueSize, true);
        /* start worker threads */

        queueDrainer = new Thread("QueueDrainer") {
            @Override
            public void run() {
                while(!exit) {
                    HttpWork work = null;
                    try {
                        while (!exit && work == null) {
                            work = workQueue.poll(1, TimeUnit.SECONDS);
                        }

                        if (work == null) {
                            continue;
                        }

                        HttpWorker w = null;
                        while (w == null) {
                            w = freeWorkers.poll();
                            if (w == null) {
                                if (activeWorkers.size() < workers) {
                                    w = spawnWorker();
                                } else {
                                    w = freeWorkers.take();
                                }
                            }
                        }
                        activeWorkers.put(w, w);
                        w.setWork(work);
                        work = null;
		    } catch (Exception e) {
		        e.printStackTrace();
		        if (work != null) {
                            work.fail(
                                "Exception catched trying to process " +
                                "sync httpserver work",
                                HttpStatus.SC_INTERNAL_SERVER_ERROR
                            );
		        }
		    }
                }
            }
        };
        queueDrainer.setDaemon(true);
    }

    public HttpWorker spawnWorker() {
        HttpWorker w = new HttpWorker(this, handler);
        Thread t = new Thread(w,
            "HttpWorker #"+id.incrementAndGet());
        t.start();
        return w;
    }


    public void start() {
        queueDrainer.start();
        for (int i = 0; i < minWorkers; ++i ) {
            HttpWorker w = spawnWorker();
            freeWorkers.add(w);
        }
    }

    public boolean offer(HttpWork work) {
        return workQueue.offer(work);
    }

    @Override
    public void close() {
        exit = true;
        while (activeWorkers.size() > 0 || freeWorkers.size() > 0) {
            Iterator<HttpWorker> iter = freeWorkers.iterator();
            while (iter.hasNext()) {
                HttpWorker w = iter.next();
                w.stop();
                iter.remove();
            }
            iter = activeWorkers.keySet().iterator();
            while (iter.hasNext()) {
                HttpWorker w = iter.next();
                w.stop();
                iter.remove();
            }
        }
    }

    public void freeWorker(HttpWorker worker) {
        activeWorkers.remove(worker);
        freeWorkers.addFirst(worker);
    }

    public boolean tryExpireWorker(HttpWorker worker) {
        if (freeWorkers.size() <= minWorkers) {
            return false;
        }
        if (!freeWorkers.remove(worker)) {
            //worker became active while trying to expire
            return false;
        } else {
            return true;
        }
    }
}
