package ru.yandex.msearch.proxy;

import java.io.IOException;

import java.nio.file.Paths;

import ru.yandex.logger.LoggersConfigBuilder;

import ru.yandex.msearch.proxy.api.Api;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.config.MsearchProxyConfigBuilder;
import ru.yandex.msearch.proxy.dispatcher.DispatcherFactory;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.mail.FolderListFactory;
import ru.yandex.msearch.proxy.ora.WmiFilterSearchClient;
import ru.yandex.msearch.proxy.parallel.ParallelExec;
import ru.yandex.msearch.proxy.search.MailSearcher;
import ru.yandex.msearch.proxy.search.Searcher;
import ru.yandex.msearch.proxy.searchmap.SearchMap;

import ru.yandex.parser.config.IniConfig;

public class Proxy {
    public static void main(final String... argv) throws Exception {
        String usage = "Proxy-config <config file name>";

        if (argv.length == 0) {
            System.out.println("msearch-proxy failed to start");
            System.out.println("Usage: " + usage);
            return;
        }
        String configFileName = null;

        for (int i = 0; i < argv.length; i++) {
            if (argv[i].equals("-config")) {
                configFileName = argv[++i];
            } else if (i != argv.length - 1) {
                System.out.println("msearch-proxy failed to start");
                System.out.println("Usage: " + usage);
                return;
            }
        }

        System.setProperty("http.maxConnections", String.valueOf(10000));
        System.setProperty("sun.net.http.errorstream.enableBuffering", "true");

        try {
            IniConfig ini = new IniConfig(Paths.get(configFileName));
            ini.renameSection("full_log", LoggersConfigBuilder.LOG);
            ini.renameSection("access_log", LoggersConfigBuilder.ACCESSLOG);

            ImmutableMsearchProxyConfig config =
                new MsearchProxyConfigBuilder(ini).build();

            Logger.init(
                config.loggers().preparedLoggers().asterisk(),
                config.errorLogConfig().build(
                    config.loggers().handlersManager()));
            Logger.info("Starting proxy");
            SearchMap.init(config.searchMapConfig().file().getCanonicalPath());
            MailSearcher.init(ini.section("mail_searcher"));
            Searcher.init(ini.section("searcher"));
            ParallelExec.init(ini);

            DispatcherFactory.init(
                ini.section("dispatcher_factory"),
                config);

            FolderListFactory.init(ini.section("folderlist"));
            WmiFilterSearchClient.init(config);

            Api api = new Api(ini.section("api"), config);
            final HttpServer server = new HttpServer(
                config,
                new Api(ini.section("api"), config));
            Runtime.getRuntime().addShutdownHook(
                new Thread() {
                    @Override
                    public void run() {
                        Logger.info("Shutting down");
                        try {
                            server.close();
                        } catch (IOException e) {
                            Logger.err("Failed to stop HttpServer");
                            e.printStackTrace();
                        }
                        Logger.deinit();
                    }
                }
            );
            ini.checkUnusedKeys();
            server.start();
            System.out.println("msearch-proxy started");
        } catch (Exception e) {
            System.out.println("msearch-proxy failed to start");
            e.printStackTrace(System.out);
            e.printStackTrace(System.err);
            Runtime.getRuntime().exit(-1);
        }
    }
}

