package ru.yandex.msearch.proxy.api;

import java.io.ByteArrayInputStream;
import java.io.IOException;

import java.nio.charset.CharacterCodingException;

import java.text.ParseException;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;

import org.apache.xerces.impl.io.UTF8Reader;

import ru.yandex.function.StringBuilderProcessorAdapter;

import ru.yandex.http.util.CharsetUtils;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.BadRequestException;
import ru.yandex.msearch.proxy.dispatcher.Dispatcher;
import ru.yandex.msearch.proxy.dispatcher.DispatcherException;
import ru.yandex.msearch.proxy.dispatcher.DispatcherFactory;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.searchmap.SearchMap;

import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;

import ru.yandex.search.json.AddDocumentsMapCollector;
import ru.yandex.search.json.AddMessageRootHandler;
import ru.yandex.search.json.HandlersManager;
import ru.yandex.search.json.MapCollector;
import ru.yandex.search.json.UpdateDocumentsMapCollector;
import ru.yandex.search.json.UpdateMessageRootHandler;

import ru.yandex.search.prefix.Prefix;

public class ServiceProcessor {
    private static final int API_LENGTH = "/api/".length();

    private final DispatchableService service;
    private final HttpServer.RequestContext ctx;
    private final String fullRequest;
    private final String request;
    private final String api;

    public ServiceProcessor(
        final DispatchableService service,
        final HttpServer.RequestContext ctx,
        final String request)
    {
        this.service = service;
        this.ctx = ctx;
        this.fullRequest = request;
        this.request =
            request.substring("/api/".length() + service.name().length());
        // remove trailing slash in /api/<name>/
        String api = this.request.substring(1);
        int idx = api.indexOf('?');
        if (idx == -1) {
            this.api = api;
        } else {
            this.api = api.substring(0, idx);
        }
    }

    public DispatchableService service() {
        return service;
    }

    public HttpServer.RequestContext ctx() {
        return ctx;
    }

    public String request() {
        return request;
    }

    public String api() {
        return api;
    }

    private void doParseJson(final JsonParser parser) throws ApiException {
        try {
            parser.parse(
                new UTF8Reader(new ByteArrayInputStream(ctx.getPostData())));
        } catch (IOException | JsonException e) {
            ctx.log.err(Logger.exception(e));
            throw new ApiException(e);
        }
    }

    private MapCollector parseJson() throws ApiException {
        HandlersManager manager = new HandlersManager();
        AddDocumentsMapCollector collector =
            new AddDocumentsMapCollector(service.prefixParser());
        manager.push(new AddMessageRootHandler(manager, collector));
        doParseJson(new JsonParser(manager));
        return collector;
    }

    private MapCollector parseJsonAndValidate()
        throws MsearchProxyException
    {
        if (ctx.getPostData() == null) {
            throw new BadRequestException(
                service.name() + ".index: empty POST");
        } else {
            MapCollector collector = parseJson();
            if (collector.getDocuments().isEmpty()) {
                throw new BadRequestException(service.name()
                    + ".index: invalid JSON format: no documents found");
            } else if (collector.getPrefix() == null) {
                throw new BadRequestException(service.name()
                    + ".index: invalid JSON format: no prefix found");
            }
            return collector;
        }
    }

    private UpdateDocumentsMapCollector parseUpdateJson() throws ApiException {
        HandlersManager manager = new HandlersManager();
        UpdateDocumentsMapCollector collector =
            new UpdateDocumentsMapCollector(service.prefixParser());
        manager.push(new UpdateMessageRootHandler(manager, collector));
        doParseJson(new JsonParser(manager));
        return collector;
    }

    private MapCollector parseUpdateJsonAndValidate()
        throws MsearchProxyException
    {
        if (ctx.getPostData() == null) {
            throw new BadRequestException(
                service.name() + ".index: error: empty POST");
        } else {
            UpdateDocumentsMapCollector collector = parseUpdateJson();
            if (collector.getDocuments().isEmpty()) {
                throw new BadRequestException(service.name()
                    + ".update: invalid JSON format: no documents found");
            } else if (collector.getPrefix() == null) {
                throw new BadRequestException(service.name()
                    + ".update: invalid JSON format: no prefix found");
            } else {
                ctx.log.debug(service.name()
                    + ".update: prefix=" + collector.getPrefix()
                    + ", prefix-hash=" + collector.getPrefix().hash()
                    + ", query=" + collector.getQuery()
                    + ", docs=" + collector.getDocuments());
            }
            return collector;
        }
    }

    private int dispatch(
        final MapCollector collector,
        final HttpRequestBase request)
        throws MsearchProxyException
    {
        return dispatch(collector.getPrefix(), request);
    }

    private int dispatch(final Prefix prefix, final HttpRequestBase request)
        throws MsearchProxyException
    {
        try {
            Dispatcher dispatcher = DispatcherFactory.create(
                prefix,
                service.serviceName(),
                SearchMap.getInstance(),
                false,
                ctx);
            dispatcher.indexDispatch(request, service.indexTimeout());
        } catch (DispatcherException e) {
            int status = HttpStatus.SC_INTERNAL_SERVER_ERROR;
            if (e instanceof DispatcherException.BadRequestException) {
                status = HttpStatus.SC_BAD_REQUEST;
            } else if (e instanceof DispatcherException.BadResponseException) {
                status = ((DispatcherException.BadResponseException) e).code();
            }
            throw new ApiException(
                service.name() + ".index: can't index documents to backend",
                e,
                status);
        }
        return 0;
    }

    public int indexDispatch(final String method, final boolean wait)
        throws MsearchProxyException
    {
        MapCollector collector = parseJsonAndValidate();
        StringBuilder sb = new StringBuilder(method);
        if (wait) {
            sb.append("?wait=true");
        }
        HttpPost post = new HttpPost(new String(sb));
        post.setEntity(
            new ByteArrayEntity(
                ctx.getPostData(),
                ContentType.APPLICATION_JSON));
        return dispatch(collector, post);
    }

    public int index(final HttpServer.HttpParams params, final boolean wait)
        throws MsearchProxyException
    {
        return indexDispatch("/add", wait);
    }

    public int modify(final HttpServer.HttpParams params, final boolean wait)
        throws MsearchProxyException
    {
        return indexDispatch("/modify", wait);
    }

    public int update(final HttpServer.HttpParams params, final boolean wait)
        throws MsearchProxyException
    {
        MapCollector collector = parseUpdateJsonAndValidate();
        StringBuilder sb = new StringBuilder("/update");
        if (wait) {
            sb.append("?wait=true");
        }
        HttpPost post = new HttpPost(new String(sb));
        post.setEntity(
            new ByteArrayEntity(
                ctx.getPostData(),
                ContentType.APPLICATION_JSON));
        return dispatch(collector, post);
    }

    public Prefix getPrefix(final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        try {
            return service.prefixParser().parse(params.get("prefix"));
        } catch (ParseException e) {
            throw new BadRequestException(
                service.name() + ": can't parse request prefix",
                e);
        }
    }

    public int delete(final HttpServer.HttpParams params, final boolean wait)
        throws MsearchProxyException
    {
        if (ctx.getPostData() == null) {
            Prefix prefix = getPrefix(params);
            String text = params.get("text");
            if (text == null) {
                throw new BadRequestException(
                    service.name() + ".delete: no query set");
            }
            ctx.log.debug(service.name() + ".delete: prefix=" + prefix
                + ", query=" + text);
            StringBuilder sb = new StringBuilder("/delete?text=");
            PctEncoder encoder = new PctEncoder(PctEncodingRule.QUERY);
            try {
                encoder.process(text.toCharArray());
                encoder.processWith(new StringBuilderProcessorAdapter(sb));
            } catch (CharacterCodingException e) {
                ctx.log.err(Logger.exception(e));
                sb.append(text);
            }
            if (wait) {
                sb.append("&wait=true");
            }
            return dispatch(prefix, new HttpGet(new String(sb)));
        } else {
            return indexDispatch("/delete", wait);
        }
    }

    public int search(final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        service.checkSearchParams(params);
        Prefix prefix = getPrefix(params);
        int timeout = params.getInt("timeout", service.searchTimeout());
        long requestStart = System.currentTimeMillis();
        Dispatcher dispatcher = DispatcherFactory.create(
            prefix,
            service.serviceName(),
            SearchMap.getInstance(),
            false,
            ctx);
        try (CloseableHttpResponse response =
                dispatcher.searchDispatch(new HttpGet(request), timeout))
        {
            int status = response.getStatusLine().getStatusCode();
            HttpEntity entity = response.getEntity();
            if (entity == null) {
                ctx.setHttpCode(status);
            } else {
                ctx.setHttpCode(status);
                if (status < HttpStatus.SC_OK
                    || status >= HttpStatus.SC_BAD_REQUEST)
                {
                    ctx.log.err(
                        service.name()
                        + ".search: server returned status: "
                        + status);
                    ctx.ps().println("Backend response: http code: "
                        + status + ", response body:");
                    String body = CharsetUtils.toString(entity);
                    ctx.log.err(
                        service.name() + ".search: server response: " + body);
                    ctx.ps().println(body);
                } else {
                    Header contentType = entity.getContentType();
                    if (contentType == null
                        || contentType.getValue() == null)
                    {
                        ctx.setContentType("application/json");
                    } else {
                        ctx.setContentType(contentType);
                    }
                    long start = System.currentTimeMillis();
                    entity.writeTo(ctx.ps());
                    ctx.log.info("Backend response read time: "
                        + (System.currentTimeMillis() - start));
                }
            }
        } catch (DispatcherException | IOException | HttpException e) {
            throw new ApiException(e, HttpStatus.SC_INTERNAL_SERVER_ERROR);
        } finally {
            ctx.log.info(
                service.name()
                + ".search: backend request execution time: "
                + (System.currentTimeMillis() - requestStart));
        }
        return 0;
    }

    public int dispatch(final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        boolean wait = service.wait(api, params);
        switch (api) {
            case "index":
            case "index_nowait":
                if (service.allowIndex()) {
                    return index(params, wait);
                }
                break;

            case "modify":
                if (service.allowModify()) {
                    return modify(params, wait);
                }
                break;

            case "update":
            case "update_nowait":
                if (service.allowUpdate()) {
                    return update(params, wait);
                }
                break;

            case "delete":
                if (service.allowDelete()) {
                    return delete(params, wait);
                }
                break;

            case "search":
                if (service.allowSearch()) {
                    return search(params);
                }

            default:
                break;
        }
        return -1;
    }

    public int unknownApi() throws MsearchProxyException {
	ctx.setHttpCode(HttpStatus.SC_NOT_IMPLEMENTED);
	ctx.ps().println("Unknown api");
	ctx.log.err("Unknown api: " + fullRequest);
	return 0;
    }
}

