package ru.yandex.msearch.proxy.api.async.broadcast;

import java.io.IOException;

import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;

import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

import ru.yandex.search.proxy.BroadcastCallback;

import ru.yandex.search.rules.RequestParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class BroadcastRule
    implements SearchRule<List<Entry<HttpHost, String>>, RequestParams, SearchInfo>
{
    private final AsyncHttpServer server;

    public BroadcastRule(final AsyncHttpServer server) {
        this.server = server;
    }

    private User buildUser(
        final CgiParams params)
        throws BadRequestException
    {
        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix prefix;
        if (mdb.equals(ProxyParams.PG)) {
            prefix = params.get(ProxyParams.UID, prefixType);
        } else {
            prefix = params.get(ProxyParams.SUID, prefixType);
        }

        return new User(server.resolveService(mdb, prefix), prefix);
    }

    @Override
    public void execute(
        final SearchRequest<List<Entry<HttpHost, String>>, RequestParams, SearchInfo> request)
        throws HttpException
    {
        SearchMapShard shard =
            server.searchMap().hosts(buildUser(request.cgiParams()));

        HttpRequest httpRequest = request.session().request();
        String path = httpRequest.getRequestLine().getUri()
            .substring("/broadcast".length());

        MultiFutureCallback<String> mfcb =
            new MultiFutureCallback<>(
                new BroadcastAggregateCallback(request.callback(), shard));

        AsyncClient client =
            server.proxyClient().adjust(request.session().context());

        try {
            for (SearchMapHost host: shard) {
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        path
                        + "&backendHost=" + host.searchHost().toHostString());

                for (Header header: httpRequest.getAllHeaders()) {
                    generator.addHeader(header);
                }

                request.session().subscribeForCancellation(
                    client.execute(
                        server.httpHost(),
                        generator,
                        AsyncStringConsumerFactory.INSTANCE,
                        new BroadcastCallback(mfcb.newCallback())));
            }
        } catch (IOException e) {
            // Quite impossible to fail server binding
            throw new ServiceUnavailableException(e);
        }

        mfcb.done();
    }

    private final class BroadcastAggregateCallback
        extends AbstractFilterFutureCallback<List<String>, List<Map.Entry<HttpHost, String>>>
    {
        private final SearchMapShard shard;

        public BroadcastAggregateCallback(
            final FutureCallback<? super List<Entry<HttpHost, String>>> cb,
            final SearchMapShard shard)
        {
            super(cb);
            this.shard = shard;
        }

        @Override
        public void completed(final List<String> strings) {
            List<Map.Entry<HttpHost, String>> result
                = new ArrayList<>(strings.size());

            for (int i = 0; i < shard.size(); i++) {
                result.add(
                    new SimpleEntry<>(
                        shard.get(i).searchHost(),
                        strings.get(i)));
            }

            callback.completed(result);
        }
    }

    private final class BroadcastCallback implements FutureCallback<String> {
        private final FutureCallback<String> callback;

        public BroadcastCallback(
            final FutureCallback<String> callback)
        {
            this.callback = callback;
        }

        @Override
        public void completed(final String s) {
            callback.completed(s);
        }

        @Override
        public void failed(final Exception e) {
            callback.completed("Failed: " + e.toString());
        }

        @Override
        public void cancelled() {
            callback.completed("Cancelled");
        }
    }
}
