package ru.yandex.msearch.proxy.api.async.enlarge;

import java.io.IOException;
import java.io.StringReader;

import java.text.ParseException;

import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.parser.config.ConfigException;

import ru.yandex.parser.uri.CgiParamsBase;
import ru.yandex.parser.uri.QueryConstructor;

public class Enlargeble {
    private final String service;
    private final List<QueryPart> queryParts;

    public Enlargeble(final String description) throws ConfigException {
        final int colon = description.indexOf(':');
        if (colon == -1) {
            throw new ConfigException(
                "Invalid Enlargeble description, "
                    + "expected: \"service:query\", but was: " + description);
        }
        this.service = description.substring(0, colon);
        try {
            queryParts = parseQuery(description.substring(colon + 1));
        } catch (ParseException e) {
            throw new ConfigException("Enlargeble query parsing failed", e);
        }
    }

    public String service() {
        return service;
    }

    private List<QueryPart> parseQuery(final String query)
        throws ParseException
    {
        final List<QueryPart> queryParts = new ArrayList<>();
        final StringBuilder partBuilder = new StringBuilder();
        final StringReader reader = new StringReader(query);
        boolean cgiPart = false;
        try {
            for (int c = reader.read(); c != -1; c = reader.read()) {
                if (c == '$') {
                    c = reader.read();
                    if (c == '{') {
                        closeCurrentPart(partBuilder, cgiPart, queryParts);
                        cgiPart = true;
                    } else {
                        partBuilder.append((char) c);
                    }
                } else if (c == '}') {
                    if (cgiPart) {
                        closeCurrentPart(partBuilder, cgiPart, queryParts);
                        cgiPart = false;
                    }
                } else {
                    partBuilder.append((char) c);
                }
            }
        } catch (IOException ignore) {
        }
        if (cgiPart) {
            throw new ParseException(
                "${CGI_PARAM} unclosed part <" + partBuilder
                    + "> in query: " + query,
                0);
        }
        closeCurrentPart(partBuilder, cgiPart, queryParts);
        if (queryParts.size() == 0) {
            throw new ParseException("Invalid or empty query: " + query, 0);
        }
        return queryParts;
    }

    private void closeCurrentPart(
        final StringBuilder sb,
        final boolean cgi,
        final List<QueryPart> queryParts)
    {
        if (sb.length() == 0) {
            return;
        }
        final String value = sb.toString();
        sb.setLength(0);
        final QueryPart part;
        if (cgi) {
            part = new CgiPart(value);
        } else {
            part = new ConstantPart(value);
        }
        queryParts.add(part);
    }

    public String apply(final CgiParamsBase params) throws HttpException {
        final QueryConstructor query = new QueryConstructor("");
        apply(query, params);
        return query.toString();
    }

    public void apply(
        final QueryConstructor query,
        final CgiParamsBase params)
        throws HttpException
    {
        for (QueryPart part: queryParts) {
            part.add(query, params);
        }
    }

    private abstract class QueryPart {
        protected final String s;

        public QueryPart(final String s) {
            this.s = s;
        }

        abstract void add(
            final QueryConstructor query,
            final CgiParamsBase params)
            throws HttpException;
    }

    private class ConstantPart extends QueryPart {
        public ConstantPart(final String s) {
            super(s);
        }

        @Override
        public void add(
            final QueryConstructor query,
            final CgiParamsBase params)
        {
            query.sb().append(s);
        }
    }

    private class CgiPart extends QueryPart {
        public CgiPart(final String name) {
            super(name);
        }

        @Override
        public void add(
            final QueryConstructor query,
            final CgiParamsBase params)
            throws BadRequestException
        {
            query.append(params.getString(s, ""));
        }
    }
}
