package ru.yandex.msearch.proxy.api.async.mail;

import java.io.IOException;

import java.net.URISyntaxException;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeSet;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotFoundException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.parser.email.types.MessageFlags;

import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.document.mail.JsonFirstlineMailMetaHandler;

public class Am22Handler implements ProxyRequestHandler {
    private static final String MID = "mid";
    private static final String MDB = ProxyParams.MDB;
    private static final String FIRST_LINE = "first_line";
    private static final String FIRSTLINE = "firstline";
    private static final String HDR_FROM = "hdr_from";
    private static final String FROM = "from";
    private static final String HDR_SUBJECT = "hdr_subject";
    private static final String SUBJECT = "subject";
    private static final String MSG_ID = "msg_id";
    private static final String ST_ID = "st_id";
    private static final String TYPES = "types";

    private final AsyncHttpServer server;
    private final String filterSearchUri;

    public Am22Handler(final AsyncHttpServer server) {
        this.server = server;
        filterSearchUri =
            server.config().filterSearchConfig().uri().toASCIIString()
            + server.config().filterSearchConfig().firstCgiSeparator()
            + "mdb=pg&full_folders_and_labels=1";
    }

    @Override
    public String toString() {
        return "Retrieves message info from database";
    }

    @Override
    public void handle(final ProxySession session) throws HttpException {
        String mid = session.params().get(MID, NonEmptyValidator.INSTANCE);
        QueryConstructor query = new QueryConstructor(filterSearchUri);
        query.append("mids", mid);
        AsyncClient client = server.filterSearchClient(false);
        try {
            client.execute(
                new AsyncGetURIRequestProducerSupplier(
                    query.toString() + "&uid="
                    + session.params().get(
                        ProxyParams.UID,
                        NonNegativeLongValidator.INSTANCE)),
                JsonAsyncTypesafeDomConsumerFactory.SPARSE_KEY_OK,
                session.listener().createContextGeneratorFor(client),
                new FilterSearchCallback(session));
        } catch (URISyntaxException e) {
            throw new BadRequestException(e);
        }
    }

    private static abstract class Am22Callback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final JsonType jsonType;

        public Am22Callback(final ProxySession session) throws HttpException {
            super(session);
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        protected abstract void writeResponse(
            final JsonWriter writer,
            final JsonMap map)
            throws HttpException, IOException, JsonException;

        @Override
        public void completed(final JsonObject root) {
            try {
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = jsonType.create(sbw)) {
                    writer.startObject();
                    writeResponse(writer, root.asMap());
                    writer.endObject();
                }
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            session.acceptedCharset())));
            } catch (HttpException e) {
                failed(e);
            } catch (IOException | JsonException e) {
                failed(new JsonException("Failed to parse: " + root, e));
            }
        }
    }

    private static class FilterSearchCallback extends Am22Callback {
        public FilterSearchCallback(final ProxySession session)
            throws HttpException
        {
            super(session);
        }

        @Override
        protected void writeResponse(
            final JsonWriter writer,
            final JsonMap map)
            throws HttpException, IOException, JsonException
        {
            JsonList envelopes = map.get("envelopes").asList();
            if (envelopes.isEmpty()) {
                throw new NotFoundException("No envelopes found");
            }
            FirstlineMailMetaInfo meta = new FirstlineMailMetaInfo();
            new JsonFirstlineMailMetaHandler(meta)
                .handle(envelopes.get(0).asMap());
            writer.key(FIRST_LINE);
            writer.value(Objects.toString(meta.firstline(), ""));
            writer.key(HDR_FROM);
            writer.value(Objects.toString(meta.get(HDR_FROM), "").trim());
            writer.key(HDR_SUBJECT);
            writer.value(Objects.toString(meta.get(HDR_SUBJECT),""));
            writer.key(MSG_ID);
            writer.value(Objects.toString(meta.get(MSG_ID), ""));
            writer.key(ST_ID);
            writer.value(Objects.toString(meta.get("stid"), ""));
            writer.key(TYPES);
            writer.value(new TreeSet<>(meta.messageTypes()));
        }
    }

    private static class MsalCallback extends Am22Callback {
        public MsalCallback(final ProxySession session) throws HttpException {
            super(session);
        }

        @Override
        protected void writeResponse(
            final JsonWriter writer,
            final JsonMap map)
            throws HttpException, IOException, JsonException
        {
            JsonList rows = map.get("rows").asList();
            if (rows.isEmpty()) {
                throw new NotFoundException("No messages found");
            }
            JsonMap message = rows.get(0).asMap();
            writer.key(FIRST_LINE);
            writer.value(message.getString(FIRSTLINE, ""));
            writer.key(HDR_FROM);
            writer.value(message.getString(FROM, ""));
            writer.key(HDR_SUBJECT);
            writer.value(message.getString(SUBJECT,""));
            writer.key(MSG_ID);
            writer.value(message.getString(MSG_ID, ""));
            writer.key(ST_ID);
            writer.value(message.getString(ST_ID, ""));
            writer.key(TYPES);
            writer.value(
                new MessageFlags(message.getLong("is_mixed")).types());
        }
    }
}

