package ru.yandex.msearch.proxy.api.async.mail;

import java.io.IOException;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.GenericConsumer;
import ru.yandex.function.GenericFunction;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.dom.IdentityContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.BasicSession;
import ru.yandex.msearch.proxy.api.async.Session;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocument;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocuments;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocumentsGroup;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.msearch.proxy.api.async.mail.result.MailFieldsPrinter;
import ru.yandex.msearch.proxy.api.async.mail.rules.PlainSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.ResolveUidRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;
import ru.yandex.msearch.proxy.api.async.mail.rules.SearchRule;

import ru.yandex.parser.string.NonEmptyValidator;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.result.BasicSearchDocument;
import ru.yandex.search.result.SearchDocument;

import ru.yandex.util.string.StringUtils;

public class LinksHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final List<String> BACKEND_FIELDS =
        Arrays.asList("url", "hid", "x_urls");

    private final AsyncHttpServer server;
    private final SearchRule rule;

    public LinksHandler(final AsyncHttpServer server) {
        this.server = server;
        rule =
            new ResolveUidRule(
                new PlainSearchRule(
                    new RuleContext(
                        server,
                        LinksDocumentsFactory.INSTANCE,
                        LinksSearchAttributesFactory.INSTANCE)),
                server);
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public String toString() {
        return "Performs links search and deduplication";
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        Session httpSession = new BasicSession(server, exchange, context);
        CgiParams params = httpSession.params();
        params.replace(
            "request",
            "x_urls:1 AND thread_id:"
            + params.get("thread_id", NonEmptyValidator.INSTANCE)
            + " AND NOT md5_p:*");
        RequestInfo requestInfo =
            new RequestInfo(httpSession, request, params);
        rule.execute(
            new BasicSearchSession(
                requestInfo,
                params,
                new LinksResultPrinter(requestInfo)));
    }

    private enum HidComparator implements Comparator<Document> {
        INSTANCE;

        private static String[] splitHid(final Document doc) {
            String hid = doc.doc().attrs().get("hid");
            if (hid == null) {
                hid = "";
            }
            return hid.split("\\.");
        }

        private static int parsePart(final String part) {
            try {
                return Integer.parseInt(part);
            } catch (NumberFormatException e) {
                return Integer.MIN_VALUE;
            }
        }

        private static int urlId(final Document doc) {
            String urlId = doc.doc().attrs().get("url_id");
            if (urlId == null) {
                return Integer.MIN_VALUE;
            } else {
                return parsePart(urlId);
            }
        }

        @Override
        public int compare(final Document lhs, final Document rhs) {
            String[] left = splitHid(lhs);
            String[] right = splitHid(rhs);
            int min = Math.min(left.length, right.length);
            int cmp = 0;
            for (int i = 0; i < min && cmp == 0; ++i) {
                cmp = Integer.compare(parsePart(left[i]), parsePart(right[i]));
            }
            if (cmp == 0) {
                cmp = Integer.compare(left.length, right.length);
            }
            if (cmp == 0) {
                cmp = Integer.compare(urlId(lhs), urlId(rhs));
            }
            if (cmp == 0) {
                cmp = lhs.id().compareTo(rhs.id());
            }
            return cmp;
        }
    }

    private static class LinksDocuments extends BasicDocuments {
        private final Set<String> urls = new HashSet<>();

        public LinksDocuments(final Comparator<Document> comparator) {
            super(comparator);
        }

        @Override
        protected DocumentsGroup createDocumentsGroup(
            final String groupId,
            final Document doc)
        {
            return new BasicDocumentsGroup(
                groupId,
                HidComparator.INSTANCE,
                doc);
        }

        private void addDoc(
            final String groupId,
            final SearchDocument doc,
            final JsonMap envelope)
        {
            String urls = doc.attrs().get("x_urls");
            String url = doc.attrs().get("url");
            if (urls != null && url != null) {
                int urlId = 0;
                for (String line: urls.split("\n")) {
                    if (!line.isEmpty()) {
                        this.urls.add(line);
                        ++urlId;
                        Map<String, String> attrs = new HashMap<>(doc.attrs());
                        attrs.put("url_id", Integer.toString(urlId));
                        attrs.put("src", line);
                        super.add(
                            groupId,
                            new BasicDocument(
                                url + '/' + urlId,
                                new BasicSearchDocument(
                                    attrs,
                                    Collections.emptyList()),
                                envelope));
                    }
                }
            }
        }

        @Override
        public void add(final String groupId, final Document doc) {
            JsonMap envelope = doc.envelope();
            SearchDocument document = doc.doc();
            addDoc(groupId, document, envelope);
            for (SearchDocument merged: document.mergedDocs()) {
                addDoc(groupId, merged, envelope);
            }
            document.mergedDocs().clear();
        }

        @Override
        public int size() {
            return urls.size();
        }

        @Override
        public void clear() {
            urls.clear();
            super.clear();
        }
    }

    private enum LinksDocumentsFactory
        implements Function<Comparator<Document>, Documents>
    {
        INSTANCE;

        @Override
        public Documents apply(final Comparator<Document> comparator) {
            return new LinksDocuments(comparator);
        }
    }

    private enum LinksSearchAttributesFactory
        implements GenericFunction<
            CgiParams,
            SearchAttributes,
            BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new SidebarSearchAttributes(params, BACKEND_FIELDS);
        }
    }

    private static class LinksResultPrinter extends ArrayResultPrinter {
        public LinksResultPrinter(final RequestInfo requestInfo) {
            super(requestInfo);
        }

        @Override
        protected void writeDocuments(
            final MailFieldsPrinter printer,
            final List<? extends DocumentsGroup> docs)
            throws IOException, JsonException
        {
            Set<Object> urls = new HashSet<>();
            urls.add(null);
            int pos = 0;
            int threshold = requestInfo.offset() + requestInfo.length();
            for (DocumentsGroup group: docs) {
                for (Document doc: group) {
                    if (pos >= threshold) {
                        return;
                    }
                    Map<String, String> attrs = doc.doc().attrs();
                    String url = attrs.get("src");
                    if (urls.add(url)) {
                        if (pos >= requestInfo.offset()) {
                            JsonMap result =
                                new JsonMap(IdentityContainerFactory.INSTANCE);
                            result.put("message", doc.envelope());
                            result.put("src", new JsonString(url));
                            printer.accept(result);
                        }
                        ++pos;
                    }
                }
            }
        }
    }
}

