package ru.yandex.msearch.proxy.api.async.mail;

import java.util.Comparator;
import java.util.regex.Pattern;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.function.GenericFunction;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.server.HttpServer;
import ru.yandex.http.util.server.LoggingServerConnection;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.BasicSession;
import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocuments;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;

import ru.yandex.msearch.proxy.api.async.mail.relevance.MtypeRequestProcessorFactory;
import ru.yandex.msearch.proxy.api.async.mail.relevance.RequestMidProcessorFactory;


import ru.yandex.msearch.proxy.api.async.mail.result.ResultPrinter;
import ru.yandex.msearch.proxy.api.async.mail.result.TopRelevantPrinter;
import ru.yandex.msearch.proxy.api.async.mail.rules.ForbiddenRequestsRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.ImapSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.MailGroupsSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.MailThreadsListRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.MailThreadsSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.MisspellRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.PersonalFactorsGatherRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.PlainSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.PureSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RankingRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.ResolveUidRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RewriteRequestRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RewriteScopeRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;
import ru.yandex.msearch.proxy.api.async.mail.rules.SearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.SelectSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.SnippetFetchSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.SoCheckRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.StoreSearchRequestRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.UserSplitRule;

import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.StaterConfigBuilder;

public class MailSearchHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    public static final String SEARCH_FILTER = "search-filter";

    private final AsyncHttpServer server;
    private final SearchRule rule;
    private final RequestsStater searchFiltersStater;

    public MailSearchHandler(
        final AsyncHttpServer server,
        final String signalPrefix)
    {
        this.server = server;

        ImmutableMsearchProxyConfig config = server.config();
        SearchRule plainSearchRule =
            new PlainSearchRule(
                new RuleContext(
                    server,
                    BasicDocumentsFactory.INSTANCE,
                    DefaultSearchAttributesFactory.INSTANCE));
        SearchRule rewriteRequestRule =
            new RewriteRequestRule(server, plainSearchRule);
        if (config.pureSearch()) {
            rewriteRequestRule = new PureSearchRule(rewriteRequestRule);
        }
        SearchRule rule = new SelectSearchRule(
            new ImapSearchRule(
                new PlainSearchRule(
                    new RuleContext(
                        server,
                        BasicDocumentsFactory.INSTANCE,
                        ImapSearchAttributesFactory.INSTANCE))),
            new SnippetFetchSearchRule(
                rewriteRequestRule,
                server.iexProxyClient(),
                config.iexProxyConfig()),
            new MailGroupsSearchRule(rewriteRequestRule),
            new MailThreadsListRule(
                rewriteRequestRule,
                new PlainSearchRule(
                    new RuleContext(
                        server,
                        BasicDocumentsFactory.INSTANCE,
                        ThreadsListSearchAttributesFactory.INSTANCE))),
            new RewriteRequestRule(
                server,
                new MailThreadsSearchRule(
                    new RuleContext(
                        server,
                        BasicDocumentsFactory.INSTANCE,
                        ThreadsSearchSearchAttributesFactory.INSTANCE))));
        if (config.erratumConfig() != null) {
            rule = new MisspellRule(rule, server.erratumClient());
        }

        if (config.indexSearchRequests()
            && server.producerStoreClient() != null)
        {
            rule = new StoreSearchRequestRule(rule, server);
        }

        if (config.soCheckConfig() != null) {
            rule = new SoCheckRule(rule, server);
        }

        rule = new PersonalFactorsGatherRule(
            rule,
            server,
            RequestMidProcessorFactory.INSTANCE,
            MtypeRequestProcessorFactory.INSTANCE);

        rule = new RankingRule(rule, server.relevance());

        if (config.userSplitConfig() != null) {
            rule = new UserSplitRule(
                rule,
                server.userSplitClient(),
                config.userSplitConfig());
        }

        rule = new ResolveUidRule(rule, server);
        Pattern forbiddenRequests = config.forbiddenRequests();
        if (forbiddenRequests != null) {
            rule = new ForbiddenRequestsRule(rule, forbiddenRequests);
        }
        this.rule = new RewriteScopeRule(rule);

        this.searchFiltersStater = server.searchFilterStater();
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public String toString() {
        return "Performs search in user mail index";
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        Session httpSession = new BasicSession(server, exchange, context);
        CgiParams params = httpSession.params();

        if (params.getString(SEARCH_FILTER, null) != null) {
            Object conn = context.getAttribute(HttpCoreContext.HTTP_CONNECTION);
            ((LoggingServerConnection) conn).setStater(searchFiltersStater);
        }

        RequestInfo requestInfo =
            new RequestInfo(httpSession, request, params);

        ResultPrinter printer;
        if (server.config().topRelevant()) {
            printer = new TopRelevantPrinter(requestInfo);
        } else {
            printer = new ResultPrinter(requestInfo);
        }

        rule.execute(new BasicSearchSession(requestInfo, params, printer));
    }

    public enum BasicDocumentsFactory
        implements Function<Comparator<Document>, Documents>
    {
        INSTANCE;

        @Override
        public Documents apply(final Comparator<Document> comparator) {
            return new BasicDocuments(comparator);
        }
    }

    private enum DefaultSearchAttributesFactory
        implements GenericFunction<
            CgiParams,
            SearchAttributes,
            BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new DefaultSearchAttributes(params);
        }
    }

    private enum ImapSearchAttributesFactory
        implements GenericFunction<
            CgiParams,
            SearchAttributes,
            BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new ImapSearchAttributes(params);
        }
    }

    private enum ThreadsListSearchAttributesFactory
        implements GenericFunction<
            CgiParams,
            SearchAttributes,
            BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new ThreadsListSearchAttributes(params);
        }
    }

    private enum ThreadsSearchSearchAttributesFactory
        implements GenericFunction<
            CgiParams,
            SearchAttributes,
            BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new ThreadsSearchSearchAttributes(params);
        }
    }
}

