package ru.yandex.msearch.proxy.api.async.mail;

import java.nio.charset.Charset;

import java.util.HashSet;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;

import ru.yandex.http.util.CharsetUtils;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;

import ru.yandex.msearch.proxy.api.async.Session;
import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;

import ru.yandex.msearch.proxy.highlight.Highlighter;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.msearch.proxy.highlight.IntervalHighlighter;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;

import ru.yandex.parser.uri.CgiParams;

// Search request invariants, i.e. params same for all subrequests
public class RequestInfo {
    public static final String PAGE = "page";
    public static final String PER_PAGE = "per_page";
    public static final String FIRST = "first";
    public static final String COUNT = "count";

    private static final int DEFAULT_COUNT = 200;

    private static final CollectionParser<String, Set<String>, Exception>
        GET_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE.andThen(String::intern),
            HashSet::new);

    private final Session session;
    private final HttpRequest request;
    private final Charset acceptedCharset;
    private final boolean hasPage;
    private final Highlighter highlighter;
    private final int offset;
    private final int length;
    private final JsonType jsonType;
    private final Set<String> get;
    private final MailSearchOptions options;

    public RequestInfo(
        final Session session,
        final HttpRequest request,
        final CgiParams params)
        throws HttpException
    {
        this.session = session;
        this.request = request;
        acceptedCharset = CharsetUtils.acceptedCharset(request);
        hasPage = params.getString(FIRST, null) == null;
        if (hasPage) {
            int page = params.get(PAGE, 0, NonNegativeIntegerValidator.INSTANCE);
            length = params.get(
                PER_PAGE,
                DEFAULT_COUNT,
                PositiveIntegerValidator.INSTANCE);
            offset = page * length;
        } else {
            offset = params.get(
                FIRST,
                0,
                NonNegativeIntegerValidator.INSTANCE);
            length = params.get(
                COUNT,
                DEFAULT_COUNT,
                NonNegativeIntegerValidator.INSTANCE);
        }

        String highlight = params.getString("highlight", null);
        if (highlight != null) {
            if ("interval".equalsIgnoreCase(highlight)) {
                this.highlighter = IntervalHighlighter.INTANCE;
            } else {
                this.highlighter = HtmlHighlighter.INSTANCE;
            }
        } else {
            this.highlighter = null;
        }
        jsonType = JsonTypeExtractor.NORMAL.extract(params);
        get = params.getAll("get", null, GET_PARSER);
        options = new MailSearchOptions(params);
    }

    public Session session() {
        return session;
    }

    public HttpRequest request() {
        return request;
    }

    public Charset acceptedCharset() {
        return acceptedCharset;
    }

    public Highlighter highlighter() {
        return highlighter;
    }

    public boolean hasPage() {
        return hasPage;
    }

    public int offset() {
        return offset;
    }

    public int length() {
        return length;
    }

    public JsonType jsonType() {
        return jsonType;
    }

    public Set<String> get() {
        return get;
    }

    public MailSearchOptions options() {
        return options;
    }
}

