package ru.yandex.msearch.proxy.api.async.mail;

import java.util.List;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.http.config.ImmutableFilterSearchConfig;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;

import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

public class SearchRequest {
    private final SortOrder sort;
    private final User user;
    private final String luceneRequest;
    private final boolean corp;
    private final ImmutableFilterSearchConfig filterSearchConfig;
    private final String filterSearchRequest;
    private final String threadsSearchRequest;
    private final long failoverSearchDelay;
    private final int requestedResultsMultipier;
    private final SearchSession session;

    public SearchRequest(
        final SortOrder sort,
        final SearchSession session,
        final RuleContext context)
        throws BadRequestException
    {
        this.sort = sort;
        this.session = session;
        CgiParams params = session.params();
        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = context.server().searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        String userIdField;
        if (mdb.equals(ProxyParams.PG)) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
            userIdField = ProxyParams.UID;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
            userIdField = ProxyParams.SUID;
        }
        corp = corp(prefix);
        ImmutableMsearchProxyConfig config = context.server().config();
        String defaultFolderSet;
        if (corp) {
            defaultFolderSet = "default";
            filterSearchConfig = config.corpFilterSearchConfig();
        } else {
            defaultFolderSet = null;
            filterSearchConfig = config.filterSearchConfig();
        }
        user = new User(context.server().resolveService(mdb, prefix), prefix);

        SearchAttributes attrs =
            context.searchAttributesFactory().apply(params);
        QueryConstructor query = new QueryConstructor(
            "/search-async-mail-search?user-id-field_disable=" + userIdField);
        query.append("user-id-term_disable", prefix.toString());
        StringBuilder get = new StringBuilder(attrs.luceneGet());
        for (String field: sort.readFields()) {
            get.append(',');
            get.append(field);
        }

        String scorer = params.getOrNull("scorer");
        if ("perfield".equals(scorer)) {
            List<String> scorerFields = params.getAll("scorer_fields");
            if (!scorerFields.isEmpty()) {
                PerfieldScorer pfScorer = new PerfieldScorer(scorerFields);
                pfScorer.applyGetFields(get);
                query.append("scorer", pfScorer.buildScorer());
            }
        }

        query.append("get", new String(get));
        query.append("prefix", prefix.toString());
        if (attrs.luceneGroup() != null) {
            query.append("group", attrs.luceneGroup());
        }

        query.append("merge_func", attrs.luceneMergeFunc());
        query.append(
            "text",
            attrs.rewriteLuceneQuery(params.getString("request")));
        query.append("service", user.service());
        query.copyIfPresent(params, "scope");
        query.copyIfPresent(params, "lowercase-expanded-terms");
        query.copyIfPresent(params, "replace-ee-expanded-terms");
        query.copyIfPresent(params, "outer");
        sort.appendTo(query);
        attrs.appendToLuceneRequest(query);
        luceneRequest = query.toString();

        StringBuilder sb =
            new StringBuilder(filterSearchConfig.uri().toASCIIString());
        sb.append(filterSearchConfig.firstCgiSeparator());
        sb.append("order=default");
        query = new QueryConstructor(sb);
        query.append(ProxyParams.MDB, mdb);
        if (uid != null) {
            query.append(ProxyParams.UID, uid.toString());
        }
        Prefix originalSuid = params.get("original-suid", suid, prefixType);
        if (originalSuid != null) {
            query.append(ProxyParams.SUID, originalSuid.toString());
        }
        String folderSet = params.getString("folder_set", defaultFolderSet);
        if (folderSet != null) {
            query.append("folder_set", folderSet);
        }
        if (attrs.onlyUnread()) {
            query.append("unread", "yes");
        }

        if (attrs.includeHiddenTrash()) {
            query.append("incl_folders", "hidden_trash");
        } else {
            if (attrs.excludeTrash()) {
                query.append("excl_folders", "trash");
            }
            if (attrs.excludeHiddenSubscriptions()) {
                query.append("excl_folders", "unsubscribe");
            }
            if (attrs.excludeSpam()) {
                query.append("excl_folders", "spam");
            }

            if (attrs.excludePending()) {
                query.append("excl_folders", "pending");
            }
            query.append("excl_folders", "hidden_trash");

        }

        attrs.appendToFilterSearchRequest(query);
        filterSearchRequest = query.toString();

        if (params.getBoolean("threaded", false)) {
            sb =
                new StringBuilder(config.threadsConfig().uri().toASCIIString());

            query = new QueryConstructor(sb);
            query.append(ProxyParams.MDB, mdb);
            if (uid != null) {
                query.append(ProxyParams.UID, uid.toString());
            }
            if (suid != null) {
                query.append(ProxyParams.SUID, suid.toString());
            }
            threadsSearchRequest = query.toString();
        } else {
            threadsSearchRequest = "";
        }

        failoverSearchDelay = params.getLongDuration(
            "failover-delay",
            config.failoverSearchDelay());

        requestedResultsMultipier = attrs.requestedResultsMultipier();
    }

    public SearchSession session() {
        return session;
    }

    public static boolean corp(final Prefix prefix) {
        if (prefix instanceof LongPrefix) {
            return BlackboxUserinfo.corp(((LongPrefix) prefix).prefix());
        } else {
            // Very, very strange case
            return false;
        }
    }

    public int requestedResultsMultipier() {
        return requestedResultsMultipier;
    }

    public User user() {
        return user;
    }

    public SortOrder sort() {
        return sort;
    }

    public String luceneRequest() {
        return luceneRequest;
    }

    public boolean corp() {
        return corp;
    }

    public ImmutableFilterSearchConfig filterSearchConfig() {
        return filterSearchConfig;
    }

    public String filterSearchRequest() {
        return filterSearchRequest;
    }

    public String threadsSearchRequest() {
        return threadsSearchRequest;
    }

    public long failoverSearchDelay() {
        return failoverSearchDelay;
    }
}

