package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.client.protocol.HttpClientContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.chemodan.AttachShieldOld;
import ru.yandex.msearch.proxy.api.chemodan.Attr;
import ru.yandex.msearch.proxy.config.ImmutableChemodanConfig;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public abstract class ChemodanContext
    implements UniversalSearchProxyRequestContext
{
    protected final User user;
    protected final long uid;
    protected final boolean corp;
    protected final String requestService;
    protected final ProxySession session;
    protected final AsyncHttpServer server;
    protected final AsyncClient searchClient;
    protected final boolean excludeSpam;
    protected final boolean excludeTrash;
    protected final Supplier<? extends HttpClientContext> contextGenerator;
    protected ChemodanFolders folders;

    protected final boolean nosid;
    private final AtomicReference<AttachShield> shieldRef
        = new AtomicReference<>(null);
    protected final boolean newSidShield;

    public ChemodanContext(
        final AsyncHttpServer server,
        final ProxySession session)
        throws BadRequestException
    {
        this.session = session;
        this.server = server;
        this.searchClient = server.searchClient().adjust(session.context());
        CgiParams params = session.params();

        uid = params.getLong(ProxyParams.UID);
        this.corp = BlackboxUserinfo.corp(uid);
        // corp ?
        if (corp) {
            this.user = new User(server.config().pgCorpQueue(), new LongPrefix(uid));
        } else {
            this.user = new User(server.config().pgQueue(), new LongPrefix(uid));
        }

        this.excludeSpam = !params.getBoolean("spam", false);
        this.excludeTrash = !params.getBoolean("trash", false);

        requestService = params.getString("service");
        contextGenerator =
            session.listener().createContextGeneratorFor(searchClient);

        this.newSidShield = params.getBoolean(
            "newSid",
            server.config().chemodanConfig().ps3183Shield());
        this.nosid = params.getBoolean("nosid", false);
    }

    @Override
    public User user() {
        return user;
    }

    @Override
    public Long minPos() {
        return null;
    }

    @Override
    public AbstractAsyncClient<?> client() {
        return searchClient;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    @Override
    public long lagTolerance() {
        return Long.MAX_VALUE;
    }

    public long uid() {
        return uid;
    }

    public String requestService() {
        return requestService;
    }

    public ProxySession session() {
        return session;
    }

    public synchronized ChemodanFolders folders() {
        return folders;
    }

    public synchronized void folders(
        final ChemodanFolders folders)
    {
        this.folders = folders;
    }

    public boolean corp() {
        return corp;
    }

    public AsyncHttpServer server() {
        return server;
    }

    public abstract Attr sortAttr();

    public boolean excludeSpam() {
        return excludeSpam;
    }

    public boolean excludeTrash() {
        return excludeTrash;
    }

    public abstract boolean desc();

    public abstract int length();

    public abstract int offset();

    public Supplier<? extends HttpClientContext> contextGenerator() {
        return contextGenerator;
    }

    public boolean nosid() {
        return nosid;
    }

    public String generateSid(
        final String stid,
        final String hid,
        final String id)
        throws HttpException
    {
        if (nosid) {
            return "";
        }

        AttachShield shield = shieldRef.get();
        if (shield == null) {
            if (newSidShield) {
                shield = new AttachShieldMailPS3183(config().webattachTtl());
                if (!server.config().chemodanConfig().ps3183Shield()) {
                    session.logger().info(
                        "Using new shield, but not storing it");
                    return shield.encode(stid, hid, id, uid);
                }

                session.logger().info(
                    "Creating and storing new version of shield PS-3183");
            } else {
                shield = new AttachShieldOld(config().webattachTtl());
            }

            if (!shieldRef.compareAndSet(null, shield)) {
                shield = shieldRef.get();
            }
        }

        return shield.encode(stid, hid, id, uid);
    }

    public String attachUrlBase() {
        if (BlackboxUserinfo.corp(uid)) {
            return config().corpWebattachUrl();
        } else {
            return config().webattachUrl();
        }
    }

    public ImmutableChemodanConfig config() {
        return server.config().chemodanConfig();
    }
}