package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class ChemodanFolders {
    private static final Map<String, String> FLD_ATTRS;
    private static final Map<String, String> FLD_META_ATTRS;

    static {
        HashMap<String, String> folderAttrs = new HashMap<>();
        folderAttrs.put("fid", "id");
        folderAttrs.put("creation_time", "ctime");
        folderAttrs.put("name", "name");

        FLD_ATTRS = Collections.unmodifiableMap(folderAttrs);
        HashMap<String, String> folderMetaAttrs = new HashMap<>();

        folderMetaAttrs.put("parent_fid", "parentId");
        folderMetaAttrs.put("parent_id", "parentId");
        folderMetaAttrs.put("special", "systemLabel");
        folderMetaAttrs.put("symbol", "systemLabel");

        FLD_META_ATTRS = Collections.unmodifiableMap(folderMetaAttrs);
    }

    private final Map<String, FolderChemodanDocument> foldersFidMap =
        new HashMap<>();
    private final Map<String, List<FolderChemodanDocument>> foldersParentMap =
        new HashMap<>();

    public ChemodanFolders(final List<Map<String,String>> folders) {
        for (Map<String, String> folder : folders) {
            FolderChemodanDocument doc = new FolderChemodanDocument();
            doc.attr("id", folder.get("id"));
            String parentId = folder.get("parent_id");
            if (parentId == null) {
                parentId = folder.get("parent_fid");
            }
            if (parentId != null) {
                doc.metaAttr("parentId", parentId);
            } else {
                parentId = "";
            }

            for (Map.Entry<String, String> entry : folder.entrySet()) {
                String attr = entry.getKey();
                String value = entry.getValue();
                String docAttr = FLD_ATTRS.get(attr);
                if (docAttr != null) {
                    doc.attr(docAttr, value);
                } else {
                    docAttr = FLD_META_ATTRS.get(attr);
                    if (docAttr != null) {
                        doc.metaAttr(docAttr, value);
                    }
                }
            }

            String system = doc.metaAttr("systemLabel");
            if (system != null
                && !system.equals("0")
                && system.length() != 0)
            {
                doc.system(true);
            }
            foldersFidMap.put(doc.id(), doc);

            List<FolderChemodanDocument> list = foldersParentMap.get(parentId);
            if (list == null) {
                list = new LinkedList<>();
                foldersParentMap.put(parentId, list);
            }
            list.add(doc);
        }
        Iterator<String> iter = foldersFidMap.keySet().iterator();
        while (iter.hasNext()) {
            String fid = iter.next();
            FolderChemodanDocument doc = foldersFidMap.get(fid);
            if (doc == null) {
                continue;
            }
            doc.metaAttr("fullname", getFullName(doc));
            doc.metaAttr("fullpath", getFullPath(doc));

            doc.metaAttr(
                "has_folders",
                foldersParentMap.containsKey(doc.id()) ? "1" : "0");
        }
    }

    public static String encodeFolderId(final String fid) {
        return "folder:" + fid;
    }

    private final String getFullPath(final FolderChemodanDocument doc) {
        String parentId = doc.parentId();
        String id = "/" + encodeFolderId(doc.id());
        if (parentId == null) {
            return id;
        }
        if (parentId.isEmpty()) {
            return id;
        }
        FolderChemodanDocument parent = foldersFidMap.get(parentId);
        if (parent == null) {
            return id;
        }
        return getFullPath(parent) + id;
    }

    private final String getFullName(final FolderChemodanDocument doc) {
        String parentId = doc.parentId();
        String name = "/" + doc.attr("name");

        if (parentId == null) {
            return name;
        }

        if (parentId.isEmpty()) {
            return name;
        }

        FolderChemodanDocument parent = foldersFidMap.get(parentId);
        if (parent == null) {
            return name;
        }

        return getFullName(parent) + name;
    }

    public Map<String, FolderChemodanDocument> foldersFidMap() {
        return foldersFidMap;
    }

    public Map<String, List<FolderChemodanDocument>> foldersParentMap() {
        return foldersParentMap;
    }
}
