package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.PathComponent;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.json.xpath.XPathContentHandler;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.parser.uri.QueryConstructor;

public class ChemodanFoldersFetcher {
    private final AsyncHttpServer proxy;
    private final ImmutableURIConfig foldersConfig;

    public ChemodanFoldersFetcher(final AsyncHttpServer proxy) {
        this.proxy = proxy;

        this.foldersConfig = proxy.config().foldersConfig();
    }

    public void fetch(
        final ChemodanContext context,
        final FutureCallback<ChemodanFolders> foldersCallback)
        throws HttpException, IOException
    {
        QueryConstructor query = new QueryConstructor(
            new StringBuilder(foldersConfig.uri().toASCIIString())
                .append(foldersConfig.firstCgiSeparator())
                .append(ProxyParams.WMI_SUFFIX));
        query.append(ProxyParams.MDB, "pg");
        query.append(ProxyParams.UID, context.uid());

        AsyncClient foldersClient = proxy.foldersClient(context.corp());

        try {
            foldersClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(query.toString()),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        context
                            .server()
                            .filterSearchTvm2Ticket(context.corp()))),
                AsyncStringConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(foldersClient),
                new InternalFoldersCallback(context, foldersCallback));
        } catch (URISyntaxException ue) {
            foldersCallback.failed(ue);
        }
    }

    private static final class InternalFoldersCallback
        extends AbstractFilterFutureCallback<String, ChemodanFolders>
    {
        private final ChemodanContext context;

        public InternalFoldersCallback(
            final ChemodanContext context,
            final FutureCallback<ChemodanFolders> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final String result) {
            context.logger().info("Folders completed " + result);
            try {
                final FoldersJsonHandler foldersHandler =
                    new FoldersJsonHandler();

                XPathContentHandler jsonHandler =
                    new XPathContentHandler(foldersHandler) {
                        @Override
                        public void endObject() throws JsonException {
                            foldersHandler.endObject();
                            super.endObject();
                        }
                    };

                JsonParser jsonParser = new JsonParser(jsonHandler);
                jsonParser.parse(result);

                ChemodanFolders folders =
                    new ChemodanFolders(foldersHandler.folderList());

                callback.completed(folders);
            } catch (JsonException je) {
                context.session().logger().log(
                    Level.WARNING,
                    "Failed to parse filter search folders response "
                        + JsonType.NORMAL.toString(result),
                    je);
                callback.failed(je);
            }
        }
    }

    private static final class FoldersJsonHandler implements PrimitiveHandler {
        private Map<String,String> folder = null;
        private List<Map<String,String>> folders =
            new ArrayList<Map<String,String>>();
        private int level = 0;

        private List<Map<String,String>> folderList() {
            return folders;
        }

        @Override
        public void handle(
            final List<PathComponent> path,
            final Object value)
        {
            if (path.size() == 3) {
                if (path.get(0).name() != null &&
                    path.get(0).name().equals("folders"))
                {
                    level = 3;
                    String fid = path.get(1).name();
                    if (fid == null) {
                        //Invalid fid
                        return;
                    }
                    if (folder == null) {
                        folder = new HashMap<String,String>();
                        folder.put("id", fid);
                    }
                    String attr = path.get(2).name();
                    if (attr == null) {
                        return;
                    }
                    if (value == null) {
                        return;
                    }
                    folder.put(attr, value.toString());
                }
            } else if (path.size() == 4) {
                level = 4;
                String attr = path.get(2).name();
                if (attr == null) {
                    return;
                }
                if (attr.equals("symbolicName")) {
                    String subAttr = path.get(3).name();
                    if (subAttr == null || value == null) {
                        return;
                    }
                    if (subAttr.equals("title") && folder != null) {
                        folder.put(attr, value.toString());
                    }
                }
            }
        }

        public void endObject() {
            if (level-- != 3) {
                return;
            }
            if (folder == null) {
                return;
            }

            String system = folder.get("symbolicName");
            if (system != null && system.length() != 0) {
                folder.put("systemFolder", "true");
                folder.put("symbol", system);
            }
            String name = folder.get("name");
            if (name.contains("|")) {
                name = name.substring(name.lastIndexOf('|') + 1);
                folder.put("name", name);
            }
            String parentId = folder.get("parentId");
            if (parentId != null && !parentId.equals("0")) {
                folder.put("parent_id", parentId);
            }

            folders.add(folder);
            folder = null;
        }
    }
}
