package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.chemodan.Attr;
import ru.yandex.msearch.proxy.api.chemodan.ChemodanMailSearcher;
import ru.yandex.msearch.proxy.document.MailSearchDocument;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.util.string.StringUtils;

public class ChemodanInfoHandler
    extends AbstractChemodanHandler
    implements ProxyRequestHandler
{
    public ChemodanInfoHandler(final AsyncHttpServer server) {
        super(server);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        ChemodanInfoContext context = new ChemodanInfoContext(server, session);
        ChemodanPrinter printer = new ChemodanPrinter(context);

        if (ChemodanUtils.isFolderPath(context.path())) {
            foldersFetcher.fetch(context, new FoldersCallback(printer, context));
        } else {
            String decodedPath = ChemodanUtils.decodeFileId(context.path());

            CollectFilesCallback searchCallback =
                new CollectFilesCallback(
                    context,
                    Collections.emptyList(),
                    true,
                    context.path(),
                    printer);

            try {
                StringBuilder url = new StringBuilder();
                url.append(context.uid());
                url.append("_");
                url.append(decodedPath);

                SearcherContext searcherContext =
                    new SearcherContext(
                        context,
                        "url:" + url.toString(),
                        10000,
                        0,
                        null,
                        Attr.NAME.luceneName(),
                        Collections.emptyList(),
                        false);

                    new ChemodanInfoSearcher(
                        searcherContext,
                        context.path(),
                        searchCallback)
                        .sendNextRequest();

//                collectFiles(
//                    context,
//                    false,
//                    Attr.ID.luceneName(),
//                    10000,
//                    0,
//                    "mid_p:" + mid,
//                    Collections.emptyList(),
//                    searchCallback);
            } catch (BadRequestException bre) {
                printer.failed(bre);
            }
        }
    }

    private class FoldersCallback
        extends AbstractFilterFutureCallback<ChemodanFolders, AsyncChemodanCollector> {
        private final ChemodanInfoContext context;
        private final String folderPath;

        public FoldersCallback(
            final FutureCallback<? super AsyncChemodanCollector> callback,
            final ChemodanInfoContext context)
            throws BadRequestException
        {
            super(callback);

            this.context = context;
            this.folderPath = ChemodanUtils.decodeFolderId(context.path());
        }

        @Override
        public void completed(final ChemodanFolders folders) {
            context.folders(folders);

            List<String> filterFids = new LinkedList<>();
            ChemodanUtils.collectFilterFids(folders, folderPath, 1, filterFids);

            CollectFoldersCallback searchCallback =
                new CollectFoldersCallback(
                    context,
                    filterFids,
                    false,
                    folderPath,
                    callback);

            try {
                collectFiles(
                    context,
                    "invalidid",
                    filterFids,
                    searchCallback);
            } catch (BadRequestException bre) {
                failed(bre);
            }
        }
    }

    private static class CollectFoldersCallback
        extends AbstractChemodanSearchCallback
    {
        private final ChemodanInfoContext context;
        private final String folderPath;

        public CollectFoldersCallback(
            final ChemodanInfoContext context,
            final Collection<String> folders,
            final boolean search,
            final String path,
            final FutureCallback<? super AsyncChemodanCollector> callback)
        {
            super(context, context.collector(), folders, search, "invalidid", callback);
            this.context = context;
            this.folderPath = path;
        }

        @Override
        public void completed(final ChemodanSearcherDocuments documents) {
            context.logger().info("Backend Search completed");
            try {
                super.fillCollector(documents);

                findFolder();

                callback.completed(collector);
            } catch (Exception e) {
                callback.failed(e);
            }
        }

        private void findFolder() {
            FolderChemodanDocument doc =
                context.folders().foldersFidMap().get(folderPath);
            if (doc == null) {
                return;
            }

            doc.metaAttr(
                "has_folders",
                context.folders().foldersParentMap().containsKey(doc.id()) ? "1" : "0");

            doc.attr("id", ChemodanUtils.encodeFolderId(doc.id()));
            context.collector().collect(doc);
        }
    }

    private static class CollectFilesCallback extends AbstractChemodanSearchCallback {
        public CollectFilesCallback(
            final ChemodanInfoContext context,
            final Collection<String> folders,
            final boolean search,
            final String path,
            final FutureCallback<? super AsyncChemodanCollector> callback)
        {
            super(context, context.collector(), folders, search, path, callback);
        }

        @Override
        public void completed(final ChemodanSearcherDocuments result) {
            context.logger().info("Backend search completed");
            super.fillCollector(result);

            callback.completed(collector);
        }
    }

    private static class ChemodanInfoContext
        extends ChemodanContext
    {
        private final String path;
        private final AsyncChemodanCollector collector;

        public ChemodanInfoContext(
            final AsyncHttpServer server,
            final ProxySession session)
            throws BadRequestException
        {
            super(server, session);
            CgiParams params = session.params();
            String path = params.getString("path", null);
            if (path == null) {
                path = params.getString("id", "");
            }

            this.collector = new AsyncChemodanCollector(Attr.NAME, true, 100);
            this.path = path;
        }

        public String path() {
            return path;
        }

        public AsyncChemodanCollector collector() {
            return collector;
        }

        @Override
        public Attr sortAttr() {
            return Attr.NAME;
        }

        @Override
        public boolean desc() {
            return false;
        }

        @Override
        public int length() {
            return 10000;
        }

        @Override
        public int offset() {
            return 0;
        }
    }

    private static class ChemodanInfoSearcher extends ChemodanSearcher {
        private final String expectedId;

        public ChemodanInfoSearcher(
            final SearcherContext searcherContext,
            final String path,
            final FutureCallback<ChemodanSearcherDocuments> callback)
            throws BadRequestException
        {
            super(searcherContext, callback);

            expectedId =
                ChemodanUtils.encodeFileId(ChemodanUtils.decodeFileId(path));
        }

        @Override
        protected void addDoc(
            final Map<String, SearchDocument> docs,
            final SearchDocument doc)
        {
            String mid = doc.attrs().get("mid");
            String hid = doc.attrs().get("hid");
            if (mid != null && hid != null) {
                String id = ChemodanMailSearcher.encodeFileId(mid + "/" + hid);
                if (id.equals(expectedId)) {
                    docs.put(mid, new MailSearchDocument(doc, mid));
                } else {
                    context.logger().info("Testing failed for id " + id + " expected id is " + expectedId);
                }
            }
        }
    }
}
