package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.text.translate.AggregateTranslator;
import org.apache.commons.lang3.text.translate.CharSequenceTranslator;
import org.apache.commons.lang3.text.translate.CodePointTranslator;
import org.apache.commons.lang3.text.translate.EntityArrays;
import org.apache.commons.lang3.text.translate.LookupTranslator;
import org.apache.http.HttpStatus;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;

public class ChemodanPrinter
    extends AbstractProxySessionCallback<AsyncChemodanCollector>
{
    private static final List<String> META_ORDER
        = Arrays.asList("fid",
        "date",
        "ImapModSeq",
        "preview",
        "hid",
        "bcc",
        "references",
        "subject",
        "part",
        "mid",
        "stid",
        "attach_size",
        "thread_id",
        "reply_to",
        "imapId",
        "tab",
        "received_date",
        "fullpath",
        "from",
        "_relevant_",
        "firstline",
        "rfcId",
        "cc",
        "hdrLastStatus",
        "types",
        "attachname",
        "inReplyTo",
        "attachsize_b",
        "uidl",
        "url",
        "revision",
        "labels",
        "st_id",
        "size",
        "extra_data",
        "new_count",
        "recv_date",
        "mimetype",
        "thread_count",
        "to",
        "fullname",
        "att_count",
        "status");
    private static final LookupTranslator
        BASIC_ESCAPE = new LookupTranslator(EntityArrays.BASIC_ESCAPE());
    private static final LookupTranslator APOS_ESCAPE =
        new LookupTranslator(EntityArrays.APOS_ESCAPE());
    private static final LookupTranslator CTRL_ESCAPE =
        new LookupTranslator(new String[][] {
            {"\n", "&#10;"},
            {"\r", "&#13;"},
            {"\t", "&#9;"}
        });

    private final ChemodanContext context;
    private final CharSequenceTranslator xmlEscaper;
    private final NonVisibleCharsTranslator badCharsEscaper;
    private StringWriter stringWriter;
    
    public ChemodanPrinter(
        final ChemodanContext context)
    {
        super(context.session());

        this.context = context;

        badCharsEscaper = new NonVisibleCharsTranslator();
        xmlEscaper = new AggregateTranslator(BASIC_ESCAPE,
            APOS_ESCAPE,
            CTRL_ESCAPE,
            badCharsEscaper);
        stringWriter = new StringWriter(1024);
    }

    @Override
    public void completed(final AsyncChemodanCollector collector) {
        StringBuilder ps = new StringBuilder();
        ps.append("<resources count=\"");
        ps.append(collector.getTotalCount());
        ps.append("\">");

        FolderChemodanDocument currentFolder;

        Map<AbstractChemodanDocument, AbstractChemodanDocument> hits = collector.hits();
        session.logger().info("Collector hits " + hits.size());
        Iterator<AbstractChemodanDocument> iter = hits.keySet().iterator();

        currentFolder = collector.getCurrentFolder();
        if (currentFolder != null) {
            currentFolder.metaAttr("CurrentFolder", "1");
            startFolder(ps, currentFolder);
        }

        int i = 0;
        while (iter.hasNext()) {
            AbstractChemodanDocument doc = iter.next();
            if (i++ < context.offset()) {
                continue;
            }
            if (i > context.length() + context.offset()) {
                break;
            }

            if (doc.folder()) {
                startFolder(ps, (FolderChemodanDocument) doc);
                endFolder(ps);
            } else {
                startFile(ps, doc);
                endFile(ps);
            }
        }
        if (currentFolder != null) {
            endFolder(ps);
        }

        session.connection().setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(i - context.offset()));

        ps.append("</resources>");
        session.response(HttpStatus.SC_OK, ps.toString());
    }

    private void startFolder(
        final StringBuilder ps,
        final FolderChemodanDocument doc)
    {
        ps.append("<folder");
        Map<String, String> attrs = doc.attrs();
        Iterator<Map.Entry<String, String>> attrIter =
            attrs.entrySet().iterator();
        while (attrIter.hasNext()) {
            Map.Entry<String, String> entry = attrIter.next();
            ps.append(" ");
            ps.append(entry.getKey());
            ps.append("=\"");
            encodeAndPrint(ps, entry.getValue());
            ps.append("\"");
        }
        ps.append(">");

        ps.append("<metainfo>");

        printMeta(ps, "filecount", Integer.toString(doc.fileCount()));
        attrs = doc.metaAttrs();
        attrIter = attrs.entrySet().iterator();
        while (attrIter.hasNext()) {
            Map.Entry<String, String> entry = attrIter.next();
            printMeta(ps, entry.getKey(), entry.getValue());
        }
        ps.append("</metainfo>");
    }

    private void printMeta(final StringBuilder ps, String key, String value)
    {
        ps.append("<meta name=\"");
        ps.append(key);
        ps.append("\" value=\"");
        encodeAndPrint(ps, value);
        ps.append("\"/>");
    }

    private void endFolder(final StringBuilder ps)
    {
        ps.append("</folder>");
    }

    private void startFile(
        final StringBuilder ps,
        final AbstractChemodanDocument doc)
    {
        List<String> orders = Arrays.asList("size", "name", "ctime", "source", "id", "type");

        ps.append("<file");
        Map<String, String> attrs = new LinkedHashMap<>(doc.attrs());
        for (String order: orders) {
            if (attrs.containsKey(order)) {
                String value = attrs.remove(order);
                ps.append(" ");
                ps.append(order);
                ps.append("=\"");
                encodeAndPrint(ps, value);
                ps.append("\"");
            }
        }

        for (Map.Entry<String, String> entry: attrs.entrySet()) {
            ps.append(" ");
            ps.append(entry.getKey());
            ps.append("=\"");
            encodeAndPrint(ps, entry.getValue());
            ps.append("\"");
        }

        ps.append(">");
        ps.append("<metainfo>");

        Map<String, String> metas = new LinkedHashMap<>(doc.metaAttrs);
        for (String name: META_ORDER) {
            if (metas.containsKey(name)) {
                String value = metas.remove(name);
                ps.append("<meta name=\"");
                ps.append(name);
                ps.append("\" value=\"");
                encodeAndPrint(ps, value);
                ps.append("\"/>");
            }
        }

        for (Map.Entry<String, String> entry: metas.entrySet()) {
            ps.append("<meta name=\"");
            ps.append(entry.getKey());
            ps.append("\" value=\"");
            encodeAndPrint(ps, entry.getValue());
            ps.append("\"/>");
        }
        ps.append("</metainfo>");
    }

    private void endFile(final StringBuilder ps)
    {
        ps.append("</file>");
    }

    private final void encodeAndPrint(final StringBuilder ps, String in)
    {
        try {
            xmlEscaper.translate(in, stringWriter);
        } catch (IOException ign) {
            //this should never happen with stringWriter
        }
        ps.append(stringWriter.toString());
        stringWriter.getBuffer().setLength(0);
        if (badCharsEscaper.hasInvalidChars()) {
            context.logger().severe(
                "ChemodanOutput: Invalid characters detected in output string: "
                    + in);
            badCharsEscaper.resetHasInvalidChars();
        }
    }

    private final String encode(String in) {
        try {
            xmlEscaper.translate(in, stringWriter);
        } catch (IOException ign) {
            //this should never happen with stringWriter
        }
        String out = stringWriter.toString();
        stringWriter.getBuffer().setLength(0);
        return out;
    }

    private class NonVisibleCharsTranslator extends CodePointTranslator {
        private boolean hasInvalidChars = false;

        public boolean hasInvalidChars()
        {
            return this.hasInvalidChars;
        }

        public void resetHasInvalidChars()
        {
            this.hasInvalidChars = false;
        }

        public boolean translate(int codepoint, Writer out) throws IOException {
            if (codepoint < ' ') {
                hasInvalidChars = true;
                out.write(' ');
                return true;
            } else {
                return false;
            }
        }
    }
}
