package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.net.URISyntaxException;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.config.FilterSearchConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;

import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;

import ru.yandex.msearch.proxy.api.async.SubrequestsCounter;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocument;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.document.MailSearchDocument;

import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

import ru.yandex.util.string.StringUtils;

public class ChemodanSearcher
    extends AbstractProxySessionCallback<SearchResult>
{
    protected final ChemodanContext context;
    private final SubrequestsCounter subrequests = new SubrequestsCounter();
    private final String luceneRequestBase;
    private final String fsRequestBase;
    private final FilterSearchConfig filterSearchConfig;
    private final AsyncClient filterSearchClient;
    private final FutureCallback<ChemodanSearcherDocuments> callback;
    private final Collector collector;
    private boolean done = false;
    private boolean hasMoreData = false;

    public ChemodanSearcher(
        final SearcherContext searcherContext,
        final FutureCallback<ChemodanSearcherDocuments> callback)
    {
        super(searcherContext.context().session());

        this.luceneRequestBase = searcherContext.searchBackendQuery();
        this.fsRequestBase = searcherContext.filterSearchQuery();
        this.context = searcherContext.context();

        if (context.corp()) {
            this.filterSearchConfig =
                context.server().config().corpFilterSearchConfig();
        } else {
            this.filterSearchConfig =
                context.server().config().filterSearchConfig();
        }

        this.filterSearchClient =
            context.server().filterSearchClient(context.corp());

        this.collector = new Collector(searcherContext.length(), searcherContext.offset());
        this.callback = callback;
    }

    protected boolean filter(final Map<String, SearchDocument> docs) {
        StringBuilder sb = new StringBuilder(fsRequestBase);
        for (String mid: docs.keySet()) {
            sb.append("&mids=");
            sb.append(mid);
        }
        try {
            filterSearchClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(new String(sb)),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        context.server().filterSearchTvm2Ticket(context.corp()))),
                JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                context.session().listener()
                    .createContextGeneratorFor(filterSearchClient),
                new FilterSearchCallback(docs));
            return true;
        } catch (URISyntaxException e) {
            failed(e);
            return false;
        }
    }

    protected void addDoc(
        final Map<String, SearchDocument> docs,
        final SearchDocument doc)
    {
        String mid = doc.attrs().get("mid");
        if (mid != null) {
            docs.put(mid, new MailSearchDocument(doc, mid));
        }
    }

    @Override
    public synchronized void completed(final SearchResult result) {
        if (done) {
            return;
        }

        context.logger().info(
            "Lucene completed " + result.host().toString() + " post "
                + result.zooQueueId());
        hasMoreData = result.hitsArray().size() >= collector.length();

        int sent = sendSubrequests(result);
        if (sent < 0) {
            return;
        }

        if (collector.requests() <= 1) {
            collector.documents().total(result.hitsCount());
        }

        if (subrequests.sent(sent)) {
            iterationCompleted();
        }
    }

    protected int sendSubrequests(final SearchResult result) {
        List<SearchDocument> hitsArray = result.hitsArray();
        int hitsCount = hitsArray.size();
        int batchSize = filterSearchConfig.batchSize();

        int subrequests = 0;
        int pos = 0;
        while (pos < hitsCount) {
            Map<String, SearchDocument> docs =
                new LinkedHashMap<>(batchSize << 1);
            while (pos < hitsCount && docs.size() < batchSize) {
                addDoc(docs, hitsArray.get(pos++));
            }

            if (hitsCount - pos < batchSize) {
                while (pos < hitsCount) {
                    addDoc(docs, hitsArray.get(pos++));
                }
            }

            if (!docs.isEmpty()) {
                if (filter(docs)) {
                    ++subrequests;
                } else {
                    return -1;
                }
            }
        }
        return subrequests;
    }

    public void subrequestDone() {
        if (subrequests.received()) {
            iterationCompleted();
        }
    }

    public void sendNextRequest() {
        String uri = collector.nextRequest();
        context.logger().info("LuceneRequest " + uri);
        context.server().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(uri),
            context.server().config().failoverSearchDelay(),
            false,
            SearchResultConsumerFactory.OK,
            context.contextGenerator(),
            this);
    }

    protected synchronized void iterationCompleted() {
        context.logger().info("Iteration completed");
        if (done) {
            return;
        }
        if (hasMoreData && collector.needMoreDocuments()) {
            sendNextRequest();
        } else {
            done = true;
            callback.completed(collector.documents());
        }
    }

    public void document(
        final String mid,
        final SearchDocument doc,
        final JsonMap envelope)
        throws JsonException
    {
        Document groupDoc = new BasicDocument(mid, doc, envelope);
        synchronized (collector) {
            collector.documents().add(mid, groupDoc);
        }
    }

    public void reduceTotalDocs(final int reducement) {
        synchronized (collector) {
            ChemodanSearcherDocuments docs = collector.documents();
            docs.total(docs.size() - reducement);
        }
    }

    private class FilterSearchCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final long startTime = System.currentTimeMillis();
        private final Map<String, SearchDocument> docs;
        private boolean done = false;

        public FilterSearchCallback(
            final Map<String, SearchDocument> docs)
        {
            super(context.session());
            this.docs = docs;
        }

        @Override
        public synchronized void completed(final JsonObject root) {
            if (done) {
                return;
            }
            StringBuilder mids = StringUtils.join(
                docs.keySet(),
                ',',
                "After filtration of mids ",
                0);
            mids.append(" the following mids left: ");
            try {
                JsonList envelopes = root.get("envelopes").asList();
                for (int i = 0; i < envelopes.size(); ++i) {
                    JsonMap envelope = envelopes.get(i).asMap();
                    String mid = envelope.getString("mid");
                    mids.append(mid);
                    mids.append(',');
                    SearchDocument doc = docs.get(mid);

                    if (doc == null) {
                        throw new JsonException(
                            "At envelope #" + i + " unexpected mid found: " + mid);
                    }

                    document(mid, doc, envelope);
                }

                int filteredOut = docs.size() - envelopes.size();

                if (filteredOut > 0) {
                    reduceTotalDocs(filteredOut);
                }
            } catch (JsonException e) {
                failed(new JsonException(
                    "Failed to parse: " + JsonType.NORMAL.toString(root), e));
                return;
            }
            mids.setLength(mids.length() - 1);
            context.logger().fine(new String(mids));
            done = true;
            subrequestDone();
        }
    }

    private class Collector {
        private final ChemodanSearcherDocuments documents;
        private volatile int nextOffset;
        private volatile int requests;
        private final int length;
        private final int offset;

        public Collector(final int length, final int offset) {
            this.offset = offset;
            this.length = length;
            nextOffset = 0;
            documents =
                new ChemodanSearcherDocuments(
                    length + offset);
        }

        public ChemodanSearcherDocuments documents() {
            return documents;
        }

        public int length() {
            return Math.max(
                filterSearchConfig.batchSize(),
                (needDocuments() - documents.size()) * 5);
        }

        public int needDocuments() {
            return length + offset;
        }

        public boolean needMoreDocuments() {
            return documents.size() < needDocuments();
        }

        public int requests() {
            return requests;
        }

        public String nextRequest() {
            StringBuilder sb = new StringBuilder(luceneRequestBase);
            sb.append("&offset=");
            sb.append(nextOffset);
            sb.append("&length=");
            int length = length();
            sb.append(length);
            nextOffset += length;
            requests += 1;
            return new String(sb);
        }
    }
}
