package ru.yandex.msearch.proxy.api.async.mail.chemodan;

import java.util.Map;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.msearch.proxy.api.ApiException;

import ru.yandex.msearch.proxy.api.chemodan.AttachShieldOld;
import ru.yandex.msearch.proxy.api.chemodan.ChemodanDocument;
import ru.yandex.msearch.proxy.api.chemodan.ChemodanMailSearcher;
import ru.yandex.msearch.proxy.config.ImmutableChemodanConfig;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.search.result.SearchDocument;

public class MailChemodanDocument extends ChemodanDocument {
    private final long uid;
    private final AttachShieldOld shield;
    private final ImmutableChemodanConfig config;
    private boolean converted;
    private SearchDocument doc;
    private SearchDocument parent;
    private ChemodanDocument folder;
    private String id;
    private String name;
    private String mid;
    private String hid;
    private String ctime;
    private String size;
    private String type;

    public MailChemodanDocument(
        final ImmutableChemodanConfig config,
        final long uid,
        final SearchDocument doc,
        final SearchDocument parent,
        final ChemodanDocument folder,
        final AttachShieldOld shield)
    {
        super(false);
        this.doc = doc;
        this.parent = parent;
        this.folder = folder;
        this.uid = uid;
        converted = false;
        this.shield = shield;
        this.config = config;
    }

    public SearchDocument getMailDocument() {
        if (parent != null) {
            return parent;
        }
        return doc;
    }

    public SearchDocument getOrigDocument()
    {
        return doc;
    }

    public void setAttr(final String key, final String value) {
        checkConverted();
        attrs.put(key, value);
    }

    public String getAttr(final String key) {
        checkConverted();
        return attrs.get(key);
    }

    public Map<String, String> getAttrs() {
        checkConverted();
        return attrs;
    }

    public final void setMetaAttr(final String key, final String value) {
        checkConverted();
        metaAttrs.put(key, value);
    }

    public final void setMetaAttrs(final ChemodanDocument other) {
        checkConverted();
        metaAttrs.putAll(other.getMetaAttrs());
    }

    public final String getMetaAttr(String key) {
        checkConverted();
        return metaAttrs.get(key);
    }

    public final Map<String, String> getMetaAttrs() {
        checkConverted();
        return metaAttrs;
    }

    private final String getHid(final SearchDocument doc) {
        String url = doc.attrs().get("url");
        int slashPos = url.indexOf('/');
        if (slashPos == -1 || slashPos >= url.length()) {
            return "0";
        }

        return url.substring(slashPos + 1);
    }

    private final void convert() {
        converted = true;

        if (ctime == null && parent != null) {
            ctime = parent.attrs().get("received_date");
        }
        if (ctime == null) {
            ctime = doc.attrs().get("received_date");
        }
        attrs.put("ctime", ctime);
        attrs.put("source", "mail");
        if (parent != null) {
            metaAttrs.putAll(parent.attrs());
        }
        metaAttrs.putAll(doc.attrs());
        if (mid == null) {
            mid = doc.attrs().get("mid");
        }
        if (hid == null) {
            hid = getHid(doc);
        }
        if (id == null) {
            id = ChemodanMailSearcher.encodeFileId(mid + "/" + hid);
        }
        if (name == null) {
            name = doc.attrs().get("attachname");
        }
        if (size == null) {
            size =
                String.valueOf(Integer.parseInt(doc.attrs().get("attachsize_b")));
        }
        if (type == null) {
            type = doc.attrs().get("mimetype");
        }

        if (folder != null) {
            metaAttrs.put(
                "fullpath",
                folder.getMetaAttr("fullpath") + "/" + id);
            metaAttrs.put(
                "fullname",
                folder.getMetaAttr("fullname") + "/" + name);
        }

        String stid;
        if (parent != null) {
            stid = parent.attrs().get("st_id");
        } else {
            stid = doc.attrs().get("st_id");
        }
        if (stid == null) {
            Logger.err("Document <" + mid + "> contains no st_id");
            stid = "1.2.3";
        }
        if (stid.startsWith("mulca:2:")) {
            stid = stid.substring(8);
        }

        attrs.put("id", id);
        attrs.put("name", name);
        attrs.put("size", size);
        attrs.put("type", type);
        String url;
        try {
            url = makeUrl(stid);
        } catch (ApiException | NumberFormatException e) {
            e.printStackTrace();
            url = "";
        }
        metaAttrs.put("url", url);
        metaAttrs.put("hid", hid);
        metaAttrs.put("stid", stid);
        metaAttrs.put("part", hid);
        if (ChemodanMailSearcher.previewableFile(this)) {
            metaAttrs.put("preview", url + "&thumb=y");
        }
    }

    private final void checkConverted() {
        if (!converted) {
            convert();
        }
    }

    @Override
    public void setOrderField(String field)
    {
        if (field.equals("name")) {
            if (name == null) {
                name = doc.attrs().get("attachname");
                attrs.put("name", name);
            }
        } else if (field.equals("id")) {
            if (mid == null) {
                mid = doc.attrs().get("mid");
            }
            if (hid == null) {
                hid = getHid(doc);
            }
            if (id == null) {
                id = ChemodanMailSearcher.encodeFileId(mid + "/" + hid);
                attrs.put("id", id);
            }
        } else if (field.equals("size")) {
            size =
                String.valueOf(Integer.parseInt(doc.attrs().get("attachsize_b")));
            attrs.put("size", size);
        } else if (field.equals("type")) {
            type = doc.attrs().get("mimetype");
            attrs.put("type", type);
        } else if (field.equals("ctime")) {
            if (ctime == null) {
                if (parent == null) {
                    ctime = doc.attrs().get("received_date");
                } else {
                    ctime = parent.attrs().get("received_date");
                }
                attrs.put("ctime", ctime);
            }
        }
        super.setOrderField(field);
    }

    public final String makeUrl(final String stid) throws ApiException {
        StringBuilder sb = new StringBuilder();
        if (BlackboxUserinfo.corp(uid)) {
            sb.append(config.corpWebattachUrl());
        } else {
            sb.append(config.webattachUrl());
        }
        sb.append(name);
        sb.append("?sid=");
        sb.append(shield.encodeSync(stid, hid, id, uid));
        sb.append("&name=");
        sb.append(name);
        sb.append("&ids=");
        sb.append(id);
        return new String(sb);
    }
}