package ru.yandex.msearch.proxy.api.async.mail.classification;

import java.io.IOException;
import java.io.OutputStreamWriter;

import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;

import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.nio.entity.NByteArrayEntity;

import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.FakeAsyncConsumer;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.HttpClientContextGenerator;

import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;

import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.prefix.LongPrefix;

import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.util.string.StringUtils;

/**
 * Checks should we show classification popup
 */
public class ClassificationHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final String GET =
        StringUtils.join(
            Arrays.asList(
                MailIndexFields.MID,
                MailIndexFields.SERP_CLICKS,
                MailIndexFields.CLICKS,
                MailIndexFields.FOLDER_TYPE,
                MailIndexFields.CLASSIFICATION_CLICK),
            ',');

    private static final EnumParser<FolderType> FOLDER_TYPE_ENUM_PARSER =
        new EnumParser<>(FolderType.class);

    private final AsyncHttpServer proxy;
    private final ImmutableClassificationConfig config;
    private final HttpClientContextGenerator httpClientContextGenerator;
    private final long lagTolerance;
    private final TimeFrameQueue<Long> errors;

    public ClassificationHandler(
        final AsyncHttpServer proxy)
        throws ConfigException
    {
        this.proxy = proxy;

        this.config = proxy.config().classificationConfig();

        RequestConfig requestConfig =
            AsyncClient.createRequestConfig(
                new HttpTargetConfigBuilder(
                    proxy.config().searchConfig()).timeout(
                        config.timeout()).build());

        httpClientContextGenerator =
            new HttpClientContextGenerator(requestConfig);
        lagTolerance = config.toleratePositions();
        errors = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
            errors,
                new NamedStatsAggregatorFactory<>(
                "classifications-errors_ammm",
                CountAggregatorFactory.INSTANCE)));
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            return new JsonAsyncTypesafeDomConsumer(
                ((HttpEntityEnclosingRequest) request).getEntity(),
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            return new FakeAsyncConsumer<>(JsonNull.INSTANCE);
        }
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context) throws HttpException, IOException
    {
        BasicProxySession session =
            new BasicProxySession(proxy, exchange, context);

        Long uid = session.params().getLong(ProxyParams.UID);

        Set<String> mids;

        if (BlackboxUserinfo.corp(uid)) {
            mids = Collections.emptySet();
        } else if (payload == JsonNull.INSTANCE) {
            // parse mids from uri
            mids =
                session.params().get(
                    "mids",
                    null,
                    new CollectionParser<>(String::trim, LinkedHashSet::new));

            if (mids == null) {
                mids = session.params().getAll(
                    "mid",
                    Collections.emptySet(),
                    new CollectionParser<>(String::trim, LinkedHashSet::new));
            }
        } else {
            try {
                JsonList midsList = payload.asList();
                mids = new LinkedHashSet<>(midsList.size());

                for (JsonObject midObj: payload.asList()) {
                    mids.add(midObj.asString());
                }
            } catch (JsonException je) {
                throw new BadRequestException("Malformed payload", je);
            }
        }

        handle(session, mids, uid);
    }

    public void handle(
        final ProxySession session,
        final Set<String> mids,
        final Long uid)
        throws HttpException, IOException
    {
        QueryConstructor query = new QueryConstructor("/search?");
        ClassificationCallback callback =
            new ClassificationCallback(session, mids);
        if (mids.size() == 0) {
            callback.completed(SearchResult.EMPTY);
            return;
        }

        try {
            query.append("prefix", uid);
            query.append("service", "change_log");
            StringBuilder sb = new StringBuilder(mids.size() * 26);
            sb.append("url:(");
            String separator = " OR ";
            for (String mid: mids) {
                sb.append(uid);
                sb.append('_');
                sb.append(mid);
                sb.append("/0");
                sb.append(separator);
            }

            if (mids.size() > 0) {
                sb.setLength(sb.length() - separator.length());
            }
            sb.append(')');

            query.append("text", sb.toString());
            query.append("get", GET);
        } catch (BadRequestException e) {
            session.logger().log(Level.SEVERE, "", e);
            callback.failed(e);
            return;
        }
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(query.toString());

        LongPrefix prefix = new LongPrefix(uid);
        User user = new User(
            AsyncHttpServer.resolveService("pg", prefix, proxy.config()),
            prefix);

        PlainUniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                user,
                null,
                lagTolerance,
                proxy.searchClient(),
                session.logger());

        proxy.parallelRequest(
            session,
            requestContext,
            generator,
            SearchResultConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                httpClientContextGenerator),
            new StatingErrorSuppressingCallback(session, errors, callback));
    }

    private static final class StatingErrorSuppressingCallback
        implements FutureCallback<SearchResult>
    {
        private final ProxySession session;
        private final FutureCallback<SearchResult> callback;
        private final TimeFrameQueue<Long> errors;

        public StatingErrorSuppressingCallback(
            final ProxySession session,
            final TimeFrameQueue<Long> errors,
            final FutureCallback<SearchResult> callback)
        {
            this.session = session;
            this.callback = callback;
            this.errors = errors;
        }

        @Override
        public void completed(final SearchResult result) {
            callback.completed(result);
        }

        protected void emptyResponse() {
            errors.accept(1L);
            session.response(
                HttpStatus.SC_OK,
                new StringEntity("[]", ContentType.APPLICATION_JSON));
        }

        @Override
        public void failed(final Exception e) {
            emptyResponse();
            session.logger().log(
                Level.WARNING,
                "Classification failed, return empty result",
                e);
        }

        @Override
        public void cancelled() {
            emptyResponse();
        }
    }
    private static final class ClassificationCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final JsonType jsonType;
        private final Charset acceptedCharset;
        private final Set<String> mids;

        public ClassificationCallback(
            final ProxySession session,
            final Set<String> mids)
            throws HttpException
        {
            super(session);

            this.mids = new LinkedHashSet<>(mids);
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            acceptedCharset = CharsetUtils.acceptedCharset(session.request());
        }

        @Override
        public void completed(final SearchResult result) {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            try (OutputStreamWriter outWriter =
                     new OutputStreamWriter(
                         out,
                         acceptedCharset.newEncoder()
                             .onMalformedInput(CodingErrorAction.REPLACE)
                             .onUnmappableCharacter(CodingErrorAction.REPLACE));
                 JsonWriter writer = jsonType.create(outWriter))
            {
                int removed = 0;
                writer.startArray();
                for (SearchDocument doc: result.hitsArray()) {
                    String mid =
                        doc.attrs().getOrDefault(
                            MailIndexFields.MID,
                            null);
                    if (mid == null) {
                        continue;
                    }
                    String selectd = doc.attrs().getOrDefault(
                        MailIndexFields.CLASSIFICATION_CLICK,
                        null);
                    if (selectd != null) {
                        session.logger().info(
                            "Filtering out, more than 2 shows "
                                + mid);
                        mids.remove(mid);
                    } else {
                        long sum = 0;
                        String clicksTotalStr =
                            doc.attrs().getOrDefault(
                                MailIndexFields.CLICKS,
                                null);
                        if (clicksTotalStr != null) {
                            sum += Long.parseLong(clicksTotalStr);
                        }

                        String serpClicksStr = doc.attrs().getOrDefault(
                            MailIndexFields.SERP_CLICKS,
                            null);
                        if (serpClicksStr != null) {
                            sum += Long.parseLong(serpClicksStr);
                        }

                        String fdTypeStr =
                            doc.attrs().get(MailIndexFields.FOLDER_TYPE);

                        FolderType folderType = FolderType.SPAM;
                        if (fdTypeStr != null) {
                            folderType =
                                FOLDER_TYPE_ENUM_PARSER.apply(fdTypeStr);
                        }

                        if (sum >= 3
                            || (folderType != FolderType.INBOX
                            && folderType != FolderType.USER))
                        {
                            removed += 1;
                            mids.remove(mid);
                        }
                    }
                }

                for (String mid: mids) {
                    writer.value(mid);
                }

                session.logger().info("Filtered out mids " + removed);

                writer.endArray();
            } catch (IOException | IllegalArgumentException e) {
                failed(e);
                return;
            }
            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                    acceptedCharset)
                    .toString());
            session.response(HttpStatus.SC_OK, entity);
        }
    }
}
