package ru.yandex.msearch.proxy.api.async.mail.dialogue;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.URISyntaxException;
import java.nio.charset.CodingErrorAction;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import org.apache.http.protocol.HttpContext;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.searcher.ProducerParallelSearcher;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class DialoguePersonalHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private final AsyncHttpServer server;

    private final String LUCENE_REQUEST =
        "&get=mid,thread_id,fid,hdr_from,hdr_to,hdr_cc,unread,received_date"
            + "&group=thread_id&sort=received_date"
            + "&text=hid:0+AND+NOT+folder_type:(trash+spam)";

    public DialoguePersonalHandler(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(server, exchange, context);

        Map<String, String> data = new HashMap<>();
        data.put("ivan.dudinov@yandex.ru", "{\n" +
            "   \"attachments\":[\n" +
            "      {\n" +
            "         \"attachname\":\"ogm240_16_3_330-3.01.prc.zip\",\n" +
            "         \"mimetype\":\"application/zip\",\n" +
            "         \"hid\":\"1.3\",\n" +
            "         \"attachsize\":318086\n" +
            "      }\n" +
            "   ],\n" +
            "   \"envelopes\":[\n" +
            "      {\n" +
            "         \"recipients\":[\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"ivan.dudinov\",\n" +
            "               \"displayName\":\"Ivan Dudinov\"\n" +
            "            },\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"durak\",\n" +
            "               \"displayName\":\"Иван Дурак\"\n" +
            "            }\n" +
            "         ],\n" +
            "         \"subject\":\"Как дела?\",\n" +
            "         \"threadId\":100500,\n" +
            "         \"threadCount\":2,\n" +
            "         \"firstline\":\"Первая строчка диалога\",\n" +
            "         \"receiveDate\":1507795509,\n" +
            "         \"unreadCount\":0\n" +
            "      }\n" +
            "   ]\n" +
            "}");

        data.put("kashei.immortal@yandex.ru", "{\n" +
            "   \"attachments\":[\n" +
            "      {\n" +
            "         \"attachname\":\"ogm240_16_3_330-3.01.prc.zip\",\n" +
            "         \"mimetype\":\"application/zip\",\n" +
            "         \"hid\":\"1.3\",\n" +
            "         \"attachsize\":318086\n" +
            "      }\n" +
            "   ],\n" +
            "   \"envelopes\":[\n" +
            "      {\n" +
            "         \"recipients\":[\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"kashei.immortal\",\n" +
            "               \"displayName\":\"Кащей Бессмертный\"\n" +
            "            },\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"durak\",\n" +
            "               \"displayName\":\"Иван Дурак\"\n" +
            "            }\n" +
            "         ],\n" +
            "         \"subject\":\"Захват мира\",\n" +
            "         \"threadId\":200500,\n" +
            "         \"threadCount\":2,\n" +
            "         \"firstline\":\"Список покупок для захвата мира\",\n" +
            "         \"receiveDate\":1507792509,\n" +
            "         \"unreadCount\":0\n" +
            "      }, \n" +
            "      {\n" +
            "            \"recipients\":[\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"durak\",\n" +
            "               \"displayName\":\"Иван Дурак\"\n" +
            "            },\n" +
            "            {\n" +
            "               \"domain\":\"yandex.ru\",\n" +
            "               \"local\":\"kashei.immortal\",\n" +
            "               \"displayName\":\"Кащей Бессмертный\"\n" +
            "            }\n" +
            "         ],\n" +
            "         \"subject\":\"Давай знакомиться\",\n" +
            "         \"threadId\":200501,\n" +
            "         \"threadCount\":2,\n" +
            "         \"firstline\":\"Меня зовут Иван Дурак, справки нет\",\n" +
            "         \"receiveDate\":1507692509,\n" +
            "         \"unreadCount\":0\n" +
            "      }\n" +
            "   ]\n" +
            "}");

        long uid = session.params().getLong(ProxyParams.UID);
        String mdb = session.params().getString(ProxyParams.MDB);
        String recipient = session.params().getString("recipient", null);
        if (recipient == null) {
            recipient = session.params().getString("dialogueId");
        }

        if (!session.params().getBoolean("nostatic", false)) {
            String responseText =
                data.get(recipient.toLowerCase(Locale.ROOT).trim());

            if (responseText == null) {
                session.response(HttpStatus.SC_OK, "[]");
            } else {
                session.response(HttpStatus.SC_OK, responseText);
            }
            return;
        }

        User user = DialoguesHandler.createUser(server, session);
        ProducerParallelSearcher searcher = new ProducerParallelSearcher(
            server, session, user);

        DialoguesRequestInfo requestInfo = new DialoguesRequestInfo(session);

        StringBuilder sb = new StringBuilder("/search?prefix=");
        sb.append(user.prefix());
        sb.append("&service=");
        sb.append(user.service());
        sb.append("&length=");
        sb.append(requestInfo.totalLength() * 20);
        sb.append(LUCENE_REQUEST);
        sb.append("+AND+(hdr_to:");
        sb.append(recipient);
        sb.append("+OR+hdr_from:");
        sb.append(recipient);
        sb.append(")");

        searcher.search(
            new BasicAsyncRequestProducerGenerator(sb.toString()),
            new DialoguesPersonalSearchCallback(
                server,
                session,
                requestInfo,
                new DialoguesPersonalCallback(
                    new DialogueCallback(session, requestInfo),
                    user,
                    searcher,
                    requestInfo)));
    }

    private final class DialoguesPersonalSearchCallback
        extends DialoguesSearchCallback
    {
        public DialoguesPersonalSearchCallback(
            final AsyncHttpServer server,
            final ProxySession session,
            final DialoguesRequestInfo requestInfo,
            final FutureCallback<List<Dialogue>> callback)
            throws BadRequestException
        {
            super(server, session, requestInfo, callback);
        }

        @Override
        protected void handleDoc(
            final SearchDocument doc,
            final Map<String, DialogueBuilder> dialogues,
            final Set<ContactParser.Email> recipients,
            final int unread)
        {
            long date =
                Long.parseLong(
                    doc.attrs().get(MailMetaInfo.RECEIVED_DATE));

            if (recipients.size() <= 0) {
                // impossible
                failed(
                    new BadRequestException(
                        "Invalid recipient size "
                            + doc.attrs().toString()
                            + '\n'
                            + doc.mergedDocs().toString()));
                return;
            }

            String mid = doc.attrs().get(MailIndexFields.MID);
            String threadId = doc.attrs().get(MailIndexFields.THREAD_ID);

            if (recipients.size() == 2) {
                DialogueBuilder db =
                    new PersonalDialogue(threadId);

                db.addRecipients(recipients);
                db.date(date);
                db.size(doc.mergedDocs().size() + 1);
                db.unreadCount(unread);
                dialogues.put(db.id(), db);
                db.mid(mid);
                db.fid(doc.attrs().get(MailIndexFields.FID));
                db.from(
                    StringUtils.strip(
                        doc.attrs().get(
                            MailIndexFields.HDR + MailIndexFields.FROM)));
            }
        }
    }

    private class PersonalDialogue extends DialogueBuilder {
        public PersonalDialogue(final String id) {
            super(id, "none");
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writeRecipients(writer);

            writer.key("threadId");
            writer.value(id());
            writer.key("subject");
            writer.value(subject());
            writer.key("firstline");
            writer.value(firstline());
            writer.key("receiveDate");
            writer.value(date());
            writer.key("unreadCount");
            writer.value(unreadCount());
            writer.key("threadCount");
            writer.value(size());
            writer.key("fid");
            writer.value(fid());
            writer.key("from");
            writer.value(from());
        }
    }

    private final class Attachment {
        private final String name;
        private final String mimetype;
        private final String hid;
        private final String size;

        public Attachment(final String name, final String mimetype, final String hid, final String size) {
            this.name = name;
            this.mimetype = mimetype;
            this.hid = hid;
            this.size = size;
        }

        public String name() {
            return name;
        }

        public String mimetype() {
            return mimetype;
        }

        public String hid() {
            return hid;
        }

        public String size() {
            return size;
        }
    }

    private final class DialogueCallback extends AbstractProxySessionCallback<Map.Entry<List<Attachment>, List<Dialogue>>>
    {
        protected final DialoguesRequestInfo requestInfo;

        public DialogueCallback(
            final ProxySession session,
            final DialoguesRequestInfo requestInfo)
        {
            super(session);

            this.requestInfo = requestInfo;
        }

        @Override
        public void completed(final Map.Entry<List<Attachment>, List<Dialogue>> data) {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            try (OutputStreamWriter outWriter =
                     new OutputStreamWriter(
                         out,
                         session.acceptedCharset().newEncoder()
                             .onMalformedInput(CodingErrorAction.REPLACE)
                             .onUnmappableCharacter(CodingErrorAction.REPLACE));
                 JsonWriter writer = requestInfo.jsonType().create(outWriter))
            {
                writer.startObject();

                writer.key("attachments");
                writer.startArray();
                for (Attachment attachment: data.getKey()) {
                    writer.startObject();
                    writer.key("attachname");
                    writer.value(attachment.name);
                    writer.key("mimetype");
                    writer.value(attachment.mimetype);
                    writer.key("hid");
                    writer.value(attachment.hid);
                    writer.key("size");
                    writer.value(attachment.size);
                    writer.endObject();
                }
                writer.endArray();

                writer.key("envelopes");
                writer.startArray();

                int index = 0;

                for (Dialogue dialogue: data.getValue()) {
                    index += 1;
                    if (index < requestInfo.offset()) {
                        continue;
                    } else if (index > requestInfo.totalLength()) {
                        break;
                    }

                    writer.startObject();
                    writer.value(dialogue);
                    writer.endObject();
                }

                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                failed(ioe);
                return;
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())
                    .toString());
            session.response(HttpStatus.SC_OK, entity);
        }

        public ProxySession session() {
            return session;
        }
    }
    private final class DialoguesPersonalCallback
        extends AbstractFilterFutureCallback<List<Dialogue>, Map.Entry<List<Attachment>, List<Dialogue>>>
    {
        private final User user;
        private final ProducerParallelSearcher searcher;
        private final DialoguesRequestInfo requestInfo;

        public DialoguesPersonalCallback(final FutureCallback<? super Map.Entry<List<Attachment>, List<Dialogue>>> callback, final User user, final ProducerParallelSearcher searcher, final DialoguesRequestInfo requestInfo) {
            super(callback);
            this.user = user;
            this.searcher = searcher;
            this.requestInfo = requestInfo;
        }

        @Override
        public void completed(final List<Dialogue> dialogues) {
            MultiFutureCallback<SearchResult> mfcb =
                new MultiFutureCallback<>(new AttachCallback(callback, dialogues, requestInfo));
            for (Dialogue dialogue: dialogues) {
                StringBuilder sb = new StringBuilder("/search?prefix=");
                sb.append(user.prefix());
                sb.append("&service=");
                sb.append(user.service());
                sb.append("&length=");
                sb.append(requestInfo.length());
                sb.append("&text=(attachname_keyword:*+AND+thread_id:");
                sb.append(dialogue.id());
                sb.append(")+AND+NOT+attachname_keyword:narod_attachment_links.html"
                        + "+AND+NOT+attachname_keyword:smime.p7s"
                        + "+AND+NOT+mimetype:application/pkcs7-signature"
                        + "+AND+NOT+headers:content-location\\:*"
                        + "+AND+NOT+headers:content-id\\:*"
                        + "&get=attachname,hid,mimetype,attachsize"
                    + "&sort=received_date");
                searcher.search(
                    new BasicAsyncRequestProducerGenerator(sb.toString()),
                    mfcb.newCallback());
            }

            mfcb.done();
        }
    }

    private final class AttachCallback
        extends AbstractFilterFutureCallback<List<SearchResult>, Map.Entry<List<Attachment>, List<Dialogue>>>
    {
        private final List<Dialogue> dialogues;
        private final DialoguesRequestInfo requestInfo;

        public AttachCallback(
            final FutureCallback<? super Map.Entry<List<Attachment>, List<Dialogue>>> callback,
            final List<Dialogue> dialogues,
            final DialoguesRequestInfo requestInfo)
        {
            super(callback);
            this.dialogues = dialogues;
            this.requestInfo = requestInfo;
        }

        @Override
        public void completed(final List<SearchResult> searchResults) {
            List<Attachment> attachments = new ArrayList<>(requestInfo.length());
            for(SearchResult searchResult: searchResults) {
                for (SearchDocument doc: searchResult.hitsArray()) {
                    attachments.add(new Attachment(
                        doc.attrs().get("mimetype"),
                        doc.attrs().get("attachname"),
                        doc.attrs().get("hid"),
                        doc.attrs().get("attachsize")));
                    if (attachments.size() >= requestInfo.length()) {
                        break;
                    }
                }
            }

            callback.completed(
                new AbstractMap.SimpleEntry<>(attachments, dialogues));
        }
    }
}
