package ru.yandex.msearch.proxy.api.async.mail.dialogue;

import java.net.URISyntaxException;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.http.config.ImmutableFilterSearchConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;

import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser;

import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class DialoguesSearchCallback
    extends AbstractFilterFutureCallback<SearchResult, List<Dialogue>>
{
    private final AsyncHttpServer server;
    private final DialoguesRequestInfo requestInfo;
    private final ProxySession session;

    public DialoguesSearchCallback(
        final AsyncHttpServer server,
        final ProxySession session,
        final DialoguesRequestInfo requestInfo,
        final FutureCallback<List<Dialogue>> callback)
        throws BadRequestException
    {
        super(callback);

        this.server = server;
        this.session = session;
        this.requestInfo = requestInfo;

    }

    private void extractRecipients(
        final String field,
        final Set<ContactParser.Email> recipients)
    {
        if (field == null) {
            return;
        }

        for (String item: field.trim().split("\n")) {
            if (!StringUtils.isEmpty(item)) {
                recipients.add(ContactParser.parseEmail(item));
            }
        }
    }

    private void extractRecipients(
        final SearchDocument doc,
        final Set<ContactParser.Email> recipients)
    {
        extractRecipients(
            doc.attrs().get(MailMetaInfo.HDR + MailMetaInfo.FROM),
            recipients);
        extractRecipients(
            doc.attrs().get(MailMetaInfo.HDR + MailMetaInfo.CC),
            recipients);
        extractRecipients(
            doc.attrs().get(MailMetaInfo.HDR + MailMetaInfo.TO),
            recipients);
    }

    protected void filterSearch(
        final Map<String, DialogueBuilder> dialogues,
        final QueryConstructor query)
        throws BadRequestException, URISyntaxException
    {
        for (DialogueBuilder dialogue: dialogues.values()) {
            query.append("mids", dialogue.mid());
        }

        AsyncClient filterSearchClient =
            server.filterSearchClient(false);

        filterSearchClient.execute(
            new HeaderAsyncRequestProducerSupplier(
                new AsyncGetURIRequestProducerSupplier(query.toString()),
                new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                    server.filterSearchTvm2Ticket(false))),
            //new AsyncGetURIRequestProducerSupplier(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
            session.listener()
                .createContextGeneratorFor(filterSearchClient),
            new DialoguesFilterSearchCallback(dialogues, callback));
    }

    protected void handleDoc(
        final SearchDocument doc,
        final Map<String, DialogueBuilder> dialogues,
        final Set<ContactParser.Email> recipients,
        final int unread)
    {
        long date =
            Long.parseLong(
                doc.attrs().get(MailMetaInfo.RECEIVED_DATE));

        if (recipients.size() <= 0) {
            // impossible
            failed(
                new BadRequestException(
                    "Invalid recipient size "
                        + doc.attrs().toString()
                        + '\n'
                        + doc.mergedDocs().toString()));
            return;
        }

        String mid = doc.attrs().get(MailMetaInfo.MID);

        if (recipients.size() > 2) {
            // ok it's thread
            String threadId = doc.attrs().get("thread_id");
            DialogueBuilder db =
                new DialogueBuilder(threadId, "thread");

            db.size(doc.mergedDocs().size() + 1);
            db.unreadCount(unread);
            db.recipients(recipients);
            db.date(date);
            db.mid(mid);
            db.fid(doc.attrs().get(MailIndexFields.FID));
            db.from(
                StringUtils.strip(
                    doc.attrs().get(
                        MailIndexFields.HDR + MailIndexFields.FROM)));
            dialogues.put(db.id(), db);
        } else {
            ContactParser.Email opponent = getOpponent(recipients);
            DialogueBuilder db = dialogues.get(opponent.address());
            if (db == null) {
                db = new DialogueBuilder(
                    opponent.address(),
                    "tetatet");

                db.addRecipients(opponent);
                db.date(date);
                db.size(doc.mergedDocs().size() + 1);
                db.unreadCount(unread);
                dialogues.put(db.id(), db);
                db.mid(mid);
                db.fid(doc.attrs().get(MailIndexFields.FID));
                db.from(
                    StringUtils.strip(
                        doc.attrs().get(
                            MailIndexFields.HDR + MailIndexFields.FROM)));
            } else {
                db.unreadCount(db.unreadCount() + unread);
                db.size(db.size() + doc.mergedDocs().size() + 1);
            }
        }
    }

    protected ContactParser.Email getOpponent(
        final Set<ContactParser.Email> recipients)
    {
        ContactParser.Email opponent = null;
        for (ContactParser.Email email : recipients) {
            String address =
                MailAliases.INSTANCE.normalizeEmail(
                    email.address());

            if (address.equalsIgnoreCase(
                requestInfo.selfEmail()))
            {
                continue;
            }

            opponent = new ContactParser.Email(email.name(), address);
        }

        if (opponent == null) {
            opponent = recipients.iterator().next();
        }

        return opponent;
    }

    @Override
    public void completed(final SearchResult result) {
        Map<String, DialogueBuilder> dialogues
            = new LinkedHashMap<>(requestInfo.totalLength());

        for (SearchDocument doc : result.hitsArray()) {
            int unread = 0;


            LinkedHashSet<ContactParser.Email> recipients =
                new LinkedHashSet<>();

            extractRecipients(doc, recipients);
            if (Boolean.parseBoolean(doc.attrs().get("unread"))) {
                unread += 1;
            }

            for (SearchDocument merged : doc.mergedDocs()) {
                extractRecipients(merged, recipients);

                if (Boolean.parseBoolean(merged.attrs().get("unread"))) {
                    unread += 1;
                }
            }

            handleDoc(doc, dialogues, recipients, unread);
        }

        ImmutableFilterSearchConfig filterSearchConfig =
            server.config().filterSearchConfig();

        StringBuilder sb =
            new StringBuilder(filterSearchConfig.uri().toASCIIString());
        sb.append(filterSearchConfig.firstCgiSeparator());
        sb.append("order=default");
        QueryConstructor query = new QueryConstructor(sb);

        try {
            query.append(
                ProxyParams.MDB,
                session.params().getString(ProxyParams.MDB));

            query.append(
                ProxyParams.UID,
                session.params().getString(ProxyParams.UID));
            filterSearch(dialogues, query);
        } catch (NumberFormatException
            | URISyntaxException
            | BadRequestException e) {
            failed(e);
        }
    }
}