package ru.yandex.msearch.proxy.api.async.mail.furita;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.GenericFunction;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.BasicSession;
import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.msearch.proxy.api.async.mail.BasicSearchSession;
import ru.yandex.msearch.proxy.api.async.mail.DefaultSearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.MailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.SearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.direct.DirectSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.direct.SimplePlainSearchRule;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.msearch.proxy.api.async.mail.result.MailFieldsPrinter;
import ru.yandex.msearch.proxy.api.async.mail.result.ResultPrinter;

import ru.yandex.msearch.proxy.api.async.mail.rules.ResolveUidRule;

import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;
import ru.yandex.msearch.proxy.api.async.mail.rules.SearchRule;
import ru.yandex.parser.uri.CgiParams;

public class FuritaHandler implements HttpAsyncRequestHandler<HttpRequest> {
    private final AsyncHttpServer server;
    private final SearchRule rule;

    public FuritaHandler(final AsyncHttpServer server) {
        this.server = server;
        this.rule = createRule(server);
    }

    private SearchRule createRule(final AsyncHttpServer server) {
        RuleContext ruleContext =
            new RuleContext(
                server,
                MailSearchHandler.BasicDocumentsFactory.INSTANCE,
                FuritaSearchAttributesFactory.INSTANCE);

        SearchRule rule = new SimplePlainSearchRule(ruleContext);
        rule = new FuritaRewriteRequestRule(rule);
        SearchRule resolveUidRule = new ResolveUidRule(rule, server);

        return new DirectSearchRule(resolveUidRule);
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        Session httpSession = new BasicSession(server, exchange, context);
        CgiParams params = new CgiParams(request);
        RequestInfo requestInfo =
            new RequestInfo(httpSession, request, params);

        FuritaResultPrinter printer = new FuritaResultPrinter(requestInfo);

        rule.execute(new BasicSearchSession(requestInfo, params, printer));
    }

    private enum FuritaSearchAttributesFactory
        implements GenericFunction<
        CgiParams,
        SearchAttributes,
        BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new FuritaSearchAttributes(params);
        }
    }

    private static class FuritaSearchAttributes
        extends DefaultSearchAttributes
    {
        private final String get;

        public FuritaSearchAttributes(
            final CgiParams params)
            throws BadRequestException
        {
            super(params);

            this.get = params.getString("get", MailIndexFields.MID);
        }

        @Override
        public String luceneGet() {
            return get;
        }
    }

    private static class FuritaResultPrinter extends ResultPrinter {
        public FuritaResultPrinter(final RequestInfo requestInfo) {
            super(requestInfo);
        }

        @Override
        protected MailFieldsPrinter extract(
            final JsonWriter writer)
        {
            if (requestInfo.get() == null || requestInfo.get().size() == 0) {
                return new FuritaSingleFieldPrinter(
                    writer,
                    MailIndexFields.MID);
            } else {
                if (requestInfo.get().size() == 1) {
                    return new FuritaSingleFieldPrinter(
                        writer,
                        requestInfo.get().iterator().next());
                } else {
                    return new FuritaFieldsPrinter(writer, requestInfo.get());
                }
            }
        }

        @Override
        protected void writeDocuments(
            final MailFieldsPrinter printer,
            final List<? extends DocumentsGroup> docs)
            throws JsonException, IOException
        {
            int docsPrinted = 0;

            for (DocumentsGroup group: docs) {
                for (Document doc: group) {
                    printer.accept(doc);
                    docsPrinted += 1;
                    if (docsPrinted > requestInfo.length()) {
                        break;
                    }
                }
            }
        }

    }

    private static class FuritaSingleFieldPrinter implements MailFieldsPrinter {
        private final JsonWriter writer;
        private final String field;

        public FuritaSingleFieldPrinter(
            final JsonWriter writer,
            final String field)
        {
            this.writer = writer;
            this.field = field;
        }

        @Override
        public void accept(final Document doc)
            throws JsonException, IOException
        {
            writer.value(doc.doc().attrs().get(field));
        }

        @Override
        public void accept(final JsonMap doc)
            throws JsonException, IOException
        {
            throw new UnsupportedOperationException();
        }
    }

    private static class FuritaFieldsPrinter implements MailFieldsPrinter {
        private final JsonWriter writer;
        private final Set<String> get;

        public FuritaFieldsPrinter(
            final JsonWriter writer,
            final Set<String> get)
        {
            this.writer = writer;
            this.get = get;
        }

        @Override
        public void accept(final Document doc)
            throws JsonException, IOException
        {
            writer.startObject();
            for (String field : get) {
                writer.key(field);
                writer.value(doc.doc().attrs().get(field));
            }
            writer.endObject();
        }

        @Override
        public void accept(final JsonMap doc)
            throws JsonException, IOException
        {
            throw new UnsupportedOperationException();
        }
    }
}
