package ru.yandex.msearch.proxy.api.async.mail.keyboard.rules;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.KeyboardStats;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.SenderInfo;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.SendersInfoConsumerFactory;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.TopWords;
import ru.yandex.msearch.proxy.api.async.mail.keyboard.TopWordsConsumerFactory;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.BackendsProvider;
import ru.yandex.search.rules.pure.providers.ProxySessionProvider;
import ru.yandex.search.rules.pure.providers.UserProvider;

public class KeyboardRule<
    T extends BackendsProvider & ProxySessionProvider & UserProvider>
    implements SearchRule<T, KeyboardStats>
{
    private static final String PREFIX = "prefix";

    private final AsyncClient client;

    public KeyboardRule(final AsyncClient client) {
        this.client = client;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super KeyboardStats> callback)
        throws HttpException
    {
        ProxySession session = input.proxySession();
        CgiParams params = session.params();
        int topSubjectWords = params.get(
            "top-subject-words",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        int topBodyWords = params.get(
            "top-body-words",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        int topContacts = params.get(
            "top-contacts",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        long fromDate = params.get(
            "from-date",
            0L,
            NonNegativeLongValidator.INSTANCE);
        long toDate = params.get(
            "to-date",
            Long.MAX_VALUE,
            NonNegativeLongValidator.INSTANCE);
        StringBuilder sb = new StringBuilder(
            "/printkeys-keyboard?&print-total-freqs&max-freq=0"
            + "&json-type=dollar");
        StringBuilder text = new StringBuilder(
            "&text=%28folder_type:sent+OR+folder_type:draft%29");
        if (fromDate != 0L || toDate != Long.MAX_VALUE) {
            text.append("+AND+received_day_p:%5B");
            text.append(fromDate);
            text.append("+TO+");
            text.append(toDate);
            text.append("%5D");
        }
        if (fromDate != 0L) {
            sb.append("&postfilter=received_date+%3e%3d+");
            sb.append(fromDate);
        }
        if (toDate != Long.MAX_VALUE) {
            sb.append("&postfilter=received_date+%3c%3d+");
            sb.append(toDate);
        }
        boolean firstInThread = params.getBoolean("first-in-thread", false);
        if (firstInThread) {
            sb.append("&dp=to-long(thread_id+tid)&postfilter=mid+equals+tid");
        }
        String prefix = input.user().prefix().toString();

        AsyncClient client = this.client.adjust(session.context());
        Supplier<? extends HttpClientContext> contextGenerator =
            session.listener().createContextGeneratorFor(client);
        DoubleFutureCallback<Map.Entry<TopWords, TopWords>, List<SenderInfo>>
            nextCallback =
                new DoubleFutureCallback<>(new SearchCallback(callback));
        DoubleFutureCallback<TopWords, TopWords> topWordsCallback =
            new DoubleFutureCallback<>(nextCallback.first());
        if (topSubjectWords == 0) {
            topWordsCallback.first().completed(new TopWords(0));
        } else {
            QueryConstructor query = new QueryConstructor(new String(sb));
            query.sb().append(text);
            query.sb().append("+AND+hid:0");
            query.sb().append("&field=hdr_subject_normalized");
            query.append(PREFIX, prefix + '#');
            query.append("user", prefix);
            client.execute(
                input.backends(),
                new BasicAsyncRequestProducerGenerator(query.toString()),
                new StatusCheckAsyncResponseConsumerFactory<>(
                    HttpStatusPredicates.OK,
                    new TopWordsConsumerFactory(topSubjectWords)),
                contextGenerator,
                topWordsCallback.first());
        }
        if (topBodyWords == 0) {
            topWordsCallback.second().completed(new TopWords(0));
        } else {
            QueryConstructor query = new QueryConstructor(new String(sb));
            query.sb().append(text);
            query.sb().append("&field=pure_body");
            query.append(PREFIX, prefix + '#');
            query.append("user", prefix);
            client.execute(
                input.backends(),
                new BasicAsyncRequestProducerGenerator(query.toString()),
                new StatusCheckAsyncResponseConsumerFactory<>(
                    HttpStatusPredicates.OK,
                    new TopWordsConsumerFactory(topBodyWords)),
                contextGenerator,
                topWordsCallback.second());
        }
        if (topContacts == 0) {
            nextCallback.second().completed(Collections.emptyList());
        } else {
            QueryConstructor sendersQuery = new QueryConstructor(
                "/search-keyboard?json-type=dollar&collector=sorted"
                + "&sort=senders_sent_count"
                + "&get=url,senders_last_contacted,senders_received_count,"
                + "senders_sent_count,senders_names&length=" + topContacts);
            sendersQuery.append(PREFIX, prefix);
            // TODO: corp support?
            sendersQuery.append("text", "senders_uid:" + prefix);
            client.execute(
                input.backends(),
                new BasicAsyncRequestProducerGenerator(
                    sendersQuery.toString()),
                SendersInfoConsumerFactory.OK,
                contextGenerator,
                nextCallback.second());
        }
    }

    private static class SearchCallback extends AbstractFilterFutureCallback<
        Map.Entry<Map.Entry<TopWords, TopWords>, List<SenderInfo>>,
        KeyboardStats>
    {
        public SearchCallback(
            final FutureCallback<? super KeyboardStats> callback)
        {
            super(callback);
        }

        @Override
        public void completed(
            final Map.Entry<Map.Entry<TopWords, TopWords>, List<SenderInfo>>
            result)
        {
            callback.completed(
                new KeyboardStats(
                    result.getKey().getKey(),
                    result.getKey().getValue(),
                    result.getValue()));
        }
    }
}

