package ru.yandex.msearch.proxy.api.async.mail.legacy.search;

import java.io.IOException;
import java.io.OutputStreamWriter;

import java.net.SocketTimeoutException;

import java.nio.charset.CodingErrorAction;

import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;

import org.apache.http.concurrent.FutureCallback;

import org.apache.http.entity.ContentType;

import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.function.GenericConsumer;

import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.GatewayTimeoutException;

import ru.yandex.http.util.nio.NByteArrayEntityFactory;

import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;

import ru.yandex.msearch.proxy.api.async.Session;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;

import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.search.result.SearchDocument;

public class LegacySearchPrinter implements FutureCallback<Documents> {
    protected final RequestInfo requestInfo;
    protected final Session session;
    protected final HttpRequest request;

    public LegacySearchPrinter(final RequestInfo requestInfo) {
        this.requestInfo = requestInfo;
        session = requestInfo.session();
        request = requestInfo.request();
    }

    @Override
    public void cancelled() {
        session.logger().warning("Request cancelled: " + request);
    }

    @Override
    public void completed(final Documents docs) {
        session.logger().info("Legacy search hits size " + docs.size());

        session.setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(docs.size()));
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        try (OutputStreamWriter outWriter =
                 new OutputStreamWriter(
                     out,
                     requestInfo.acceptedCharset().newEncoder()
                         .onMalformedInput(CodingErrorAction.REPLACE)
                         .onUnmappableCharacter(CodingErrorAction.REPLACE));
             JsonWriter writer = requestInfo.jsonType().create(outWriter))
        {
            writer.startObject();
            writer.key("hitsCount");
            writer.value(docs.total());
            writer.key("hitsArray");
            writer.startArray();
            Set<String> get = requestInfo.get();

            GenericConsumer<Map<String, String>, IOException> fieldsPrinter;
            if (get == null) {
                fieldsPrinter = new AllFieldsPrinter(writer);
            } else {
                fieldsPrinter = new SelectedFieldsPrinter(writer, get);
            }

            for (DocumentsGroup dg: docs.sort()) {
                SearchDocument doc = dg.best().doc();
                writer.startObject();

                fieldsPrinter.accept(doc.attrs());

                if (!doc.mergedDocs().isEmpty()) {
                    writer.key("merged_docs");
                    writer.startArray();
                    for (SearchDocument sd: doc.mergedDocs()) {
                        writer.startObject();
                        fieldsPrinter.accept(sd.attrs());
                        writer.endObject();
                    }
                    writer.endArray();
                }

                writer.endObject();
            }

            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            failed(e);
            return;
        }

        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                requestInfo.acceptedCharset())
                .toString());
        session.response(HttpStatus.SC_OK, entity);
    }

    // CSOFF: FinalParameters
    @Override
    public void failed(Exception e) {
        session.logger().log(
            Level.WARNING,
            "Request failed: " + request + '\n'
                + session.requestsListener().details(),
            e);
        if (e instanceof IOException) {
            if (e instanceof SocketTimeoutException) {
                e = new GatewayTimeoutException(e);
            } else {
                e = new BadGatewayException(e);
            }
        }
        session.handleException(e);
    }
    // CSON: FinalParameters

    public static class SelectedFieldsPrinter
        implements GenericConsumer<Map<String, String>, IOException>
    {
        private final JsonWriter writer;
        private final Set<String> names;

        public SelectedFieldsPrinter(
            final JsonWriter writer,
            final Set<String> names)
        {
            this.writer = writer;
            this.names = names;
        }

        @Override
        public void accept(final Map<String, String> doc) throws IOException {
            for (String name: names) {
                String field = doc.getOrDefault(name, null);

                if (field != null) {
                    writer.key(name);
                    writer.value(field);
                }
            }
        }
    }

    public static class AllFieldsPrinter
        implements GenericConsumer<Map<String, String>, IOException>
    {
        private final JsonWriter writer;

        public AllFieldsPrinter(final JsonWriter writer) {
            this.writer = writer;
        }

        @Override
        public void accept(final Map<String, String> doc) throws IOException {
            for (Map.Entry<String, String> entry: doc.entrySet()) {
                writer.key(entry.getKey());
                writer.value(entry.getValue());
            }
        }
    }
}