package ru.yandex.msearch.proxy.api.async.mail.ml;

import java.io.IOException;
import java.io.OutputStreamWriter;

import java.net.SocketTimeoutException;
import java.nio.charset.CodingErrorAction;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import java.util.logging.Level;

import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.GatewayTimeoutException;

import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;

import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;

public class MLResultPrinter implements FutureCallback<Documents> {
    protected final RequestInfo requestInfo;
    protected final Session session;
    protected final HttpRequest request;

    public MLResultPrinter(
        final RequestInfo requestInfo)
    {
        this.requestInfo = requestInfo;
        session = requestInfo.session();
        request = requestInfo.request();
    }

    @Override
    public void cancelled() {
        session.logger().warning("Request cancelled: " + request);
    }

    // CSOFF: FinalParameters
    @Override
    public void failed(Exception e) {
        session.logger().log(
            Level.WARNING,
            "Request failed: " + request + '\n'
            + session.requestsListener().details(),
            e);
        if (e instanceof IOException) {
            if (e instanceof SocketTimeoutException) {
                e = new GatewayTimeoutException(e);
            } else {
                e = new BadGatewayException(e);
            }
        }
        session.handleException(e);
    }
    // CSON: FinalParameters

    protected void prologue(final JsonWriter writer, final Documents docs)
        throws IOException
    {
        writer.startObject();
        writeDetails(writer, docs);
        writer.key("results");
        writer.startArray();
    }

    protected void writeDocuments(
        final JsonWriter writer,
        final Documents docs)
        throws JsonException, IOException
    {
        prologue(writer, docs);
        writeDocuments(docs.sort(), writer);
        epilogue(writer);
    }

    protected void writeDocuments(
        final List<? extends DocumentsGroup> docs,
        final JsonWriter writer)
        throws JsonException, IOException
    {
        int pos = 0;

        int threshold = requestInfo.offset() + requestInfo.length();
        for (DocumentsGroup group: docs) {
            for (Document doc: group) {
                if (pos >= threshold) {
                    return;
                }

                if (pos >= requestInfo.offset()) {
                    writeDocument(writer, (MlDocumentsGroup) group, doc);
                }
                ++pos;
            }
        }
    }

    protected void writeDocument(
        final JsonWriter writer,
        final MlDocumentsGroup group,
        final Document doc)
        throws JsonException, IOException
    {
        writer.startObject();

        writer.key("snippet");
        writer.startObject();

        writer.key("id");
        writer.value(doc.id());

        writer.key("thread_id");
        JsonObject threadId = doc.envelope().get("threadId");
        writer.value(threadId);
        if (threadId == null) {
            session.logger().info("Null thread id for doc " + JsonType.NORMAL.toString(doc.envelope()));
        }

        writer.key("title");
        writer.value(doc.envelope().get("subject"));

        writer.key("time");
        writer.value(doc.envelope().get("date"));

        writer.key("author");
        JsonObject author = null;
        try {
            author = doc.envelope().get("from").asList().get(0).get("local");
        } catch (JsonException e) {
            author = new JsonString("anonymous");
        }
        writer.value(author);

        writer.key("maillist");
        writer.value(group.mailing().name());

        writer.key("text");
        writer.value(doc.envelope().get("firstline"));

//        session.logger().info(
//            "Printing " + group.mailing().toString());
//        writer.key("envelope");
//        doc.envelope().writeValue(writer);

        writer.endObject();

        writer.endObject();
    }

    protected void epilogue(final JsonWriter writer) throws IOException {
        writer.endArray();
        writer.endObject();
    }

    @Override
    public void completed(final Documents docs) {
        session.logger().info(
            "Request completed: " + request
                + ", SERP size: " + docs.size());

        session.setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(docs.size()));
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        try (OutputStreamWriter outWriter =
                new OutputStreamWriter(
                    out,
                    requestInfo.acceptedCharset().newEncoder()
                        .onMalformedInput(CodingErrorAction.REPLACE)
                        .onUnmappableCharacter(CodingErrorAction.REPLACE));
            JsonWriter writer = requestInfo.jsonType().create(outWriter))
        {
            // TODO: make group.sort() return Collection<Collection<Documents>>
            // or something
            writeDocuments(writer, docs);
        } catch (IOException | JsonException e) {
            failed(e);
            return;
        }
        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                requestInfo.acceptedCharset())
                .toString());
        session.response(HttpStatus.SC_OK, entity);
    }

    protected void writeDetails(final JsonWriter writer, final Documents docs)
        throws IOException
    {
        writer.key("meta");
        writer.startObject();

        writer.key("page");
        writer.value(requestInfo.offset() / requestInfo.length());
        writer.key("per_page");
        writer.value(requestInfo.length());
        writer.key("next");
        writer.value((String) null);

        writer.key("doc_count");
        writer.value(docs.total());
        writer.key("pages_count");
        writer.value(docs.total() / requestInfo.length());

        writer.key("maillists");
        writer.startArray();
        for (DocumentsGroup group: docs) {
            MlDocumentsGroup mlGroup = (MlDocumentsGroup) group;
            writer.startObject();
            writer.key("name");
            writer.value(mlGroup.mailing().name());
            writer.key("count");
            writer.value(mlGroup.size());
            writer.endObject();
        }
        writer.endArray();

        writer.endObject();
    }
}

