package ru.yandex.msearch.proxy.api.async.mail.ml;

import java.io.File;
import java.io.FileReader;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.search.prefix.LongPrefix;

public class MailingParser {
    private static final MailingWeight DEFAULT_WEIGHT = new MailingWeight(0, 0);
    private final Map<Long, MailingWeight> openMailingsWeights;
    private final AsyncHttpServer server;

    public MailingParser(
        final AsyncHttpServer server,
        final PrefixedLogger logger)
    {
        this.server = server;
        Map<Long, MailingWeight> weights = Collections.emptyMap();
        try {
            JsonList list = TypesafeValueContentHandler.parse(
                new FileReader(
                    new File(
                    "ml_search_static_factors.conf"),
                    StandardCharsets.UTF_8)).asList();
            weights = new LinkedHashMap<>();
            for (JsonObject itemObj: list) {
                JsonMap map = itemObj.asMap();
                Long uid = map.getLong("uid");
                int openRate = map.getInt("open_rate", 0);
                int lastMailAndPeopleSenderRate = map.getInt("mailtype_lastmes_rank", 0);
                weights.put(uid, new MailingWeight(openRate, lastMailAndPeopleSenderRate));
            }
            logger.info("Corp ml weights loaded " + weights.size());
        } catch (Exception e) {
            logger.log(Level.WARNING, "Failed to parse mail corp mailings weights", e);
        }

        this.openMailingsWeights = weights;
    }

    public Mailing parse(final JsonMap map) throws JsonException {
        return parse(map, DEFAULT_WEIGHT);
    }

    public Mailing parse(final JsonMap map, final MailingWeight weight) throws JsonException {
        long id = map.getLong("id");
        long uid = map.getLong("uid");
        String name = map.getString("name");
        String email = map.getString("email");
        LongPrefix prefix = new LongPrefix(uid);
        return new Mailing(id, prefix, name, email, server.resolveService("pg", prefix), weight);
    }

    public Mailing parseWithWeight(final JsonMap map) throws JsonException {
        return parseWithWeight(map, 0);
    }

    public Mailing parseWithWeight(final JsonMap map, final int bonusWeight) throws JsonException {
        long id = map.getLong("id");
        long uid = map.getLong("uid");

        MailingWeight weight = openMailingsWeights.getOrDefault(uid, DEFAULT_WEIGHT);
        if (bonusWeight != 0) {
            weight = new MailingWeight(weight.openRate() + bonusWeight, weight.lastMailAndPeopleSenderRate());
        }

        String name = map.getString("name");
        String email = map.getString("email");
        LongPrefix prefix = new LongPrefix(uid);
        return new Mailing(id, prefix, name, email, server.resolveService("pg", prefix), weight);
    }
}
