package ru.yandex.msearch.proxy.api.async.mail.ml;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocument;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.BasicSearchDocument;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.util.string.StringUtils;

public class MailingsSearchRule {
    private static String TMPL_SUBS_UIDS = "%SUBSCRIPTIONS%";
    private static String TMPL_DEADLINE = "%DEADLINE%";
    private static String TMPL_QUERY = "%QUERY_BASE%";
    private static String TMPL_MAX_LENGTH = "%MAX_LENGTH%";
    private static List<String> SEARCH_FIELDS
        = Arrays.asList("pure_body", "hdr_from", "hdr_to", "hdr_cc", "hdr_subject");

    private final AsyncHttpServer server;
    private final String script;

    public MailingsSearchRule(
        final AsyncHttpServer server)
        throws IOException
    {
        this.server = server;
        this.script = IOStreamUtils.consume(
            new InputStreamReader(
                AsyncHttpServer.class.getResourceAsStream("ml_search_request.js"),
                StandardCharsets.UTF_8))
            .toString();
    }

    private QueryConstructor buildRequest(
        final SearchRequestText queryText,
        final SearchSession session)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("get", "mid");
//            qc.append("group", "mid");
//            qc.append("merge_func", "none");
        qc.append("length", session.requestInfo().length());

        qc.append("text", "");
        queryText.fieldsQuery(
            qc.sb(),
            SEARCH_FIELDS);
        qc.sb().append("+AND+hid:0");
        return qc;
    }

    private class MlSearchContext
        implements UniversalSearchProxyRequestContext
    {
        private final User user;
        private final AsyncClient client;
        private final ProxySession session;

        public MlSearchContext(final ProxySession session, final Long uid) {
            this.session = session;
            this.user = new User(server.config().pgCorpQueue(), new LongPrefix(uid));
            this.client = server.searchClient().adjust(session.context());
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }
    }

    public void execute(
        final MlSearchSession session,
        final UserMailings mailings)
        throws HttpException
    {
        PrefixedLogger logger = session.logger();

        String requestText = session.params().getString("request");
        Set<String> mailingsNameFilter = session.params().get(
            "maillist",
            Collections.emptySet(),
            new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new));

        SearchRequestText.SearchCollector collector =
            new SearchRequestText.SearchCollector(
                SearchRequestText.DEFAULT_WORDS_MODIFIER,
                SearchRequestText.DEFAULT_PHRASES_MODIFIER);

        SearchRequestText.parse(SearchRequestText.normalize(requestText), collector);

        SearchRequestText queryText = new SearchRequestText(
            requestText,
            collector.words(),
            collector.negations(),
            collector.phrases(),
            Collections.emptySet());

        int wordNum = 0;
        Map<Integer, Mailing> matchedByWord = new LinkedHashMap<>();
        Set<Long> matchedByWordIds = new LinkedHashSet<>();
        String[] split = requestText.split("\\s+");
        if (split.length > 1) {
            for (int i = 0; i < split.length; i++) {
                String word = split[i];
                Mailing mailing = mailings.get(word.trim().toLowerCase(Locale.ENGLISH));
                if (mailing != null) {
                    mailing.weight(
                        new MailingWeight(
                            MailingWeight.MAX_WEIGHT.openRate(),
                            MailingWeight.MAX_WEIGHT.lastMailAndPeopleSenderRate() - wordNum));
                    matchedByWord.put(i, mailing);
                    matchedByWordIds.add(mailing.uid());
                }

                wordNum += 1;
            }
        }

        Collection<Mailing> mailingsCol = mailings.values();
        if (!mailingsNameFilter.isEmpty()) {
            mailingsCol = new ArrayList<>(mailingsNameFilter.size());
            for (String name: mailingsNameFilter) {
                Mailing mailing = mailings.get(name);
                if (mailing != null) {
                    mailingsCol.add(mailing);
                }
            }

            logger.info("Searching only in " + mailingsCol);
        }
        if (mailingsCol.isEmpty()) {
            logger.warning("No subscriptions, returning empty result");
            session.callback().completed(new MlSearchDocuments());
            return;
        }


        Map<List<HttpHost>, List<Mailing>> map = new LinkedHashMap<>();
        for (Mailing item: mailingsCol) {
            if (!matchedByWordIds.contains(item.uid())) {
                map.computeIfAbsent(server.searchMap().searchHosts(item), (k) -> new ArrayList<>()).add(item);
            }
        }

        MailingsSearchRule.AggregateCallback aggregator =
            new AggregateCallback(session, mailings, matchedByWord.size() + map.size());

        Long deadline =
            session.params().getLong(
                "deadline",
                System.currentTimeMillis() + server.config().searchConfig().timeout());

        if (matchedByWord.size() > 0) {
            logger.info("Matched by word: " + matchedByWord);

            for (Map.Entry<Integer, Mailing> entry: matchedByWord.entrySet()) {
                String word = split[entry.getKey()];
                Mailing mailing = entry.getValue();

                MlSearchContext context = new MlSearchContext(session.session(), mailing.uid());

                split[entry.getKey()] = "";
                String sepRequest = StringUtils.join(split, ' ');
                split[entry.getKey()] = word;
                
                QueryConstructor qc = new QueryConstructor("/search?");
                qc.append("get", "mid");
                qc.append("length", session.requestInfo().length());
                qc.append("prefix", mailing.uid());
                qc.append("service", server.config().pgCorpQueue());
                SearchRequestText sepRequestText = new SearchRequestText(sepRequest);
                StringBuilder textSb = new StringBuilder();
                sepRequestText.fieldsQuery(textSb, SEARCH_FIELDS);
                textSb.append(" AND hid:0");
                qc.append("text", textSb.toString());
                server.sequentialRequest(
                    session.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    (System.currentTimeMillis() - deadline) / 2,
                    false,
                    JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                    session.session().listener().createContextGeneratorFor(context.client()),
                    new SearchBackendCallback(aggregator, mailing.uid(), session));
            }
        }

        QueryConstructor qc;
        try {
            qc = buildRequest(queryText, session);
        } catch (BadRequestException e) {
            session.callback().failed(e);
            return;
        }

        logger.info("Request is " + qc.toString());

        String script = this.script.replaceAll(TMPL_QUERY, qc.toString());
        script = script.replaceAll(
            TMPL_MAX_LENGTH,
            String.valueOf(session.requestInfo().length()));

        script = script.replaceAll(TMPL_DEADLINE, String.valueOf(deadline));

        try {
            AsyncClient client = server.searchClient().adjust(session.session().context());

            logger.info("Total inums " + map.size());

            for (Map.Entry<List<HttpHost>, List<Mailing>> inum : map.entrySet()) {
                int uidsCount = 0;
                StringBuilder uidsArray = new StringBuilder();
                uidsArray.append('[');

                Collections.sort(inum.getValue(), MailingWeightComparator.INSTANCE);

                for (Mailing mailing: inum.getValue()) {
                    uidsArray.append('\"');
                    uidsArray.append(mailing.uid());
                    uidsArray.append('\"');
                    uidsArray.append(',');
                    uidsCount += 1;
                }
                uidsArray.setLength(uidsArray.length() - 1);
                uidsArray.append(']');

                String inumScript = script.replaceAll(TMPL_SUBS_UIDS, uidsArray.toString());

                logger.info(inum.getKey().toString() + " uids count " + uidsCount);
                if (uidsCount > 0) {
                    logger.info(
                        inum.getKey().toString()
                            + " first mailings "
                            + inum.getValue().subList(0, Math.min(inum.getValue().size() - 1, 15)));
                }

                List<HttpHost> hosts;
                if (inum.getKey().size() >= 2) {
                    hosts = inum.getKey().subList(0, 2);
                } else {
                    hosts = Collections.singletonList(inum.getKey().iterator().next());
                }
                session.session().subscribeForCancellation(
                    client.execute(
                        hosts,
                        new BasicAsyncRequestProducerGenerator(
                            "/execute",
                            new StringEntity(inumScript, StandardCharsets.UTF_8)),
                        deadline,
                        (System.currentTimeMillis() - deadline) / 2,
                        JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                        session.session().listener()
                            .createContextGeneratorFor(client),
                        new SearchBackendCallback(aggregator, hosts, session)));
            }
        } catch (IOException e) {
            session.callback().failed(e);
            return;
        }
    }

    private static class AggregateCallback implements FutureCallback<List<MailingResult>> {
        private final MlSearchSession session;
        private final MlSearchDocuments result;
        private final UserMailings mailings;
        private final FutureCallback<? super Documents> callback;
        private final AtomicInteger left;
        private volatile boolean done = false;

        public AggregateCallback(
            final MlSearchSession session,
            final UserMailings mailings,
            final int requests)
        {
            this.mailings = mailings;
            this.callback = session.callback();
            this.session = session;
            this.left = new AtomicInteger(requests);

            result = new MlSearchDocuments();
            //documents = new ArrayList<>();
        }

        @Override
        public void completed(final List<MailingResult> results) {
            boolean done = false;
            int added = 0;
            synchronized (this) {
                if (!this.done) {
                    for (MailingResult mr: results) {
                        result.add(mailings.get(mr.uid()), mr.documents());
                        added += mr.documents().size();
                    }
                }
            }

            session.logger().info(
                "Backend request finished with docs: " + added + " done: " + done);

            if (done) {
                callback.completed(result);
            } else if (left.decrementAndGet() <= 0) {
                callback.completed(result);
            } else {
                session.logger().info("Requests left " + left.get());
            }
        }

        @Override
        public void cancelled() {
            session.logger().warning("Backend request canceled");
            if (!done && left.decrementAndGet() <= 0) {
                Documents result;
                synchronized (this) {
                    result = this.result;
                }

                callback.completed(result);
            }
        }

        @Override
        public void failed(final Exception e) {
            session.logger().log(Level.WARNING, "Backend request failed", e);
            if (!done && left.decrementAndGet() <= 0) {
                done = true;
                Documents result;
                synchronized (this) {
                    result = this.result;
                }

                session.logger().info(
                    "No more requests left, sending result " + result.size());
                callback.completed(result);
            } else {
                session.logger().info("Requests left " + left.get());
            }
        }
    }

    protected void filterSearch(
        final Long uid,
        final MlSearchSession session,
        final Map<String, SearchDocument> docs,
        final FutureCallback<? super MailingResult> callback)
    {
        StringBuilder sb = new StringBuilder(server.config().corpFilterSearchConfig().uri().toASCIIString());
        sb.append(server.config().corpFilterSearchConfig().firstCgiSeparator());
        sb.append("order=default");
        sb.append("&folder_set=default");
        sb.append("&uid=");
        sb.append(uid);
        sb.append("&mdb=pg");
        for (String mid: docs.keySet()) {
            sb.append("&mids=");
            sb.append(mid);
        }

        //sb.setLength(sb.length() - 1);

        try {
            AsyncClient filterSearchClient = server.filterSearchClient(true);
            session.session().subscribeForCancellation(
                filterSearchClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(new String(sb)),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(true))),
                    JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                    session.session().listener()
                        .createContextGeneratorFor(filterSearchClient),
                    new FilterSearchCallback(callback, uid, docs, session)));
        } catch (URISyntaxException e) {
            callback.failed(e);
        }
    }

    private static class FilterSearchCallback extends AbstractFilterFutureCallback<JsonObject, MailingResult> {
        private final Map<String, SearchDocument> docs;
        private final MlSearchSession session;
        private final Long uid;

        public FilterSearchCallback(
            final FutureCallback<? super MailingResult> callback,
            final Long uid,
            final Map<String, SearchDocument> docs,
            final MlSearchSession session)
        {
            super(callback);
            this.docs = docs;
            this.session = session;
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject root) {
            List<Document> result;
            StringBuilder mids = new StringBuilder();
            try {
                JsonList envelopes = root.get("envelopes").asList();
                result = new ArrayList<>(envelopes.size());

                for (int i = 0; i < envelopes.size(); ++i) {
                    JsonMap envelope = envelopes.get(i).asMap();
                    String mid = envelope.getString("mid");
                    mids.append(mid);
                    mids.append(',');
                    SearchDocument doc = docs.get(mid);
                    if (doc == null) {
                        throw new JsonException(
                            "At envelope #" + i + " unexpected mid found: " + mid + " expected " + docs.keySet());
                    }

                    Document document = new BasicDocument(mid, doc, envelope);
                    result.add(document);
                }

                int filteredOut = docs.size() - envelopes.size();

                if (filteredOut > 0) {
                    session.logger().info("Filtered out docs " + filteredOut);
                }

                callback.completed(new MailingResult(uid, result));
            } catch (JsonException e) {
                failed(new JsonException(
                    "Failed to parse: " + JsonType.NORMAL.toString(root), e));
                return;
            }
        }
    }


    private class SearchBackendCallback
        extends AbstractFilterFutureCallback<JsonObject, List<MailingResult>>
    {
        private final MlSearchSession searchSession;
        private final List<HttpHost> hosts;
        private final Long uid;

        public SearchBackendCallback(
            final FutureCallback<? super List<MailingResult>> callback,
            final List<HttpHost> hosts,
            final MlSearchSession searchSession)
        {
            this(callback, null, hosts, searchSession);
        }

        private SearchBackendCallback(
            final FutureCallback<? super List<MailingResult>> callback,
            final Long uid,
            final MlSearchSession searchSession)
        {
            this(callback, uid, Collections.emptyList(), searchSession);
        }

        private SearchBackendCallback(
            final FutureCallback<? super List<MailingResult>> callback,
            final Long uid,
            final List<HttpHost> hosts,
            final MlSearchSession searchSession)
        {
            super(callback);

            this.uid = uid;
            this.hosts = hosts;
            this.searchSession = searchSession;
        }

        private int parseAndFilterDocs(
            final Long uid,
            final JsonList backendDocs,
            final MultiFutureCallback<MailingResult> mfcb)
            throws JsonException
        {
            int count = 0;
            Map<String, SearchDocument> docs = new LinkedHashMap<>();
            for (JsonObject doc: backendDocs) {
                JsonMap docMap = doc.asMap();
                Map<String, String> attrs = new LinkedHashMap<>(docMap.size());
                for (Map.Entry<String, JsonObject> entry: docMap.entrySet()) {
                    attrs.put(entry.getKey(), entry.getValue().asString());
                }

                count += 1;
                String mid = docMap.getString("mid");
                docs.put(mid, new BasicSearchDocument(attrs, Collections.emptyList()));
            }

            if (docs.size() > 0) {
                filterSearch(uid, searchSession, docs, mfcb.newCallback());
            }

            return count;
        }

        protected void parseSingleUidResponse(
            MultiFutureCallback<MailingResult> mfcb,
            final JsonObject backendResult)
            throws JsonException
        {
            parseAndFilterDocs(uid, backendResult.asMap().getList("hitsArray"), mfcb);
        }

        protected void parseExecuteResponse(
            MultiFutureCallback<MailingResult> mfcb,
            final JsonObject backendResult)
            throws JsonException
        {
            JsonMap map = backendResult.asMap();
//            searchSession.httpSession().logger().info(
//                "Search backend completed with " + JsonType.NORMAL.toString(backendResult));
            if (map.size() == 0) {
                callback.completed(Collections.emptyList());
                return;
            }

            JsonMap result = map.getMap("result");
            int count = 0;
            for (Map.Entry<String, JsonObject> uidResult: result.entrySet()) {
                Long uid = Long.parseLong(uidResult.getKey());
                JsonList backendDocs = uidResult.getValue().asList();
                count += parseAndFilterDocs(uid, backendDocs, mfcb);
            }

            int uidsProcessed = map.getInt("uids_processed", -1);
            int uidsTotal = map.getInt("uids_total", -1);
            String lastUid = map.getString("last_uid", "none");
            searchSession.httpSession().logger().info(
                "Search backend  " + hosts + " returned docs " + count
                    + " processed mailings " + uidsProcessed + '/' + uidsTotal + " last processed " + lastUid);
        }

        @Override
        public void completed(final JsonObject backendResult) {
            MultiFutureCallback<MailingResult> mfcb = new MultiFutureCallback<>(callback);
            try {
                if (uid != null) {
                    parseSingleUidResponse(mfcb, backendResult);
                } else {
                    parseExecuteResponse(mfcb, backendResult);
                }
                mfcb.done();
            } catch (JsonException | NumberFormatException e) {
                failed(e);
                return;
            }
        }
    }

    private static class MailingResult {
        private final Long uid;
        private final List<Document> documents;

        public MailingResult(final Long uid, final List<Document> documents) {
            this.uid = uid;
            this.documents = documents;
        }

        public Long uid() {
            return uid;
        }

        public List<Document> documents() {
            return documents;
        }
    }
}
