package ru.yandex.msearch.proxy.api.async.mail.relevance;

import java.util.ArrayList;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.Product;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.factors.LuceneFieldFactorFactory;

import ru.yandex.msearch.proxy.config.ImmutableFactorsLogConfig;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.msearch.proxy.config.ImmutableRelevanceConfig;

import ru.yandex.msearch.proxy.experiment.UserSplit;

import ru.yandex.msearch.proxy.logger.ProxyTskvLogger;

import ru.yandex.parser.config.ConfigException;

import ru.yandex.parser.uri.CgiParams;

public abstract class AbstractRelevanceFactory<T extends SorterFactory<?>, D>
    implements RelevanceFactory<T>
{
    protected static final String DEFAULT = "default";
    protected static final String DISABLED = "disabled_";

    protected final Set<String> textFields = new LinkedHashSet<>();
    protected final ProxyTskvLogger factorsLogger;
    protected final ImmutableMsearchProxyConfig proxyConfig;
    protected final ImmutableFactorsLogConfig logFactorsConfig;
    protected final T defaultFactory;
    protected final Map<String, Map<String, Map<String, T>>> sortersFactories;
    protected final Map<String, Object> status;

    protected abstract Map<String, FactorGroupFactory<D>> factors();

    public AbstractRelevanceFactory(
        final AsyncHttpServer server,
        final Collection<? extends ImmutableRelevanceConfig> modelConfigs)
        throws ConfigException
    {
        this.proxyConfig = server.config();
        this.factorsLogger = server.tskvLogger();
        this.logFactorsConfig = proxyConfig.tskvLogConfig();
        this.defaultFactory = buildDefaultFactory();
        this.sortersFactories = build(modelConfigs, defaultFactory);
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("status", "ACTIVE");
        this.status = Collections.unmodifiableMap(status);

        server.logger().info(
            "Current relevance: " + String.valueOf(this.sortersFactories));
    }

    public T defaultFactory() {
        return defaultFactory;
    }

    public ProxyTskvLogger factorsLogger() {
        return factorsLogger;
    }

    public ImmutableMsearchProxyConfig proxyConfig() {
        return proxyConfig;
    }

    public Set<String> textFields() {
        return textFields;
    }

    protected abstract T createDefaultFactory(
        final List<FactorGroupFactory<D>> factors,
        final List<FactorGroupFactory<D>> logFactors)
        throws ConfigException;

    protected abstract T createModelFactory(
        final ImmutableRelevanceConfig modelConfig,
        final List<FactorGroupFactory<D>> factors,
        final List<FactorGroupFactory<D>> logFactors)
        throws ConfigException;

    protected T buildDefaultFactory() throws ConfigException {
        T defaultFactory;
        if (logFactorsConfig != null) {
            defaultFactory = createDefaultFactory(
                Collections.emptyList(),
                logFactors(logFactorsConfig));

        } else {
            defaultFactory =
                createDefaultFactory(
                    Collections.emptyList(),
                    Collections.emptyList());
        }

        return defaultFactory;
    }

    protected Map<String, Map<String, Map<String, T>>> build(
        final Collection<? extends ImmutableRelevanceConfig> configs,
        final T defaultFactory)
        throws ConfigException
    {
        Map<String, Map<String, Map<String, T>>> sorterFactories =
            new LinkedHashMap<>();

        for (ImmutableRelevanceConfig rankConfig : configs) {
            List<FactorGroupFactory<D>> factories = new ArrayList<>();
            for (String factorName : rankConfig.factors()) {
                if (!factors().containsKey(factorName)) {
                    throw new ConfigException("Unknown factor " + factorName);
                }

                FactorGroupFactory<D> factory = factors().get(factorName);
                // TODO do smth with this bullshit, or it will be coming
                // back every night in your dreams
                if (factory instanceof LuceneFieldFactorFactory) {
                    textFields.add(
                        ((LuceneFieldFactorFactory) factory).field());
                }

                factories.add(factory);
            }

            List<FactorGroupFactory<D>> logFactories = new ArrayList<>();
            if (logFactorsConfig != null) {
                logFactories = logFactors(logFactorsConfig);
            }

            Set<String> sides =
                rankConfig.sides().stream().map(Side::name)
                    .collect(Collectors.toSet());
            if (sides.isEmpty()) {
                sides = Collections.singleton(DEFAULT);
            }

            Set<String> products =
                rankConfig.products().stream().map(Product::name)
                    .collect(Collectors.toSet());

            if (products.isEmpty()) {
                products = Collections.singleton(DEFAULT);
            }

            String expId;
            switch (rankConfig.usageStatus()) {
                case DEFAULT:
                    expId = DEFAULT;
                    break;
                case EXPERIMENT:
                    expId = rankConfig.testId();
                    break;
                default:
                    expId = DISABLED + rankConfig.name();
            }

            for (String side : sides) {
                for (String product: products) {
                    sorterFactories.computeIfAbsent(
                        side,
                        (k) -> new LinkedHashMap<>())
                        .computeIfAbsent(
                            product,
                            (p) -> new LinkedHashMap<>())
                        .put(
                            expId,
                            createModelFactory(
                                rankConfig,
                                factories,
                                logFactories));
                }
            }
        }

        sorterFactories.computeIfAbsent(
            DEFAULT,
            (k) -> new LinkedHashMap<>());

        sorterFactories.forEach(
            (k, v) -> v.computeIfAbsent(
                DEFAULT,
                (p) -> new LinkedHashMap<>()));

        sorterFactories.values().forEach(
            (p) -> p.values().forEach(
                (e) -> e.putIfAbsent(DEFAULT, defaultFactory)));

        return sorterFactories;
    }

    protected List<FactorGroupFactory<D>> logFactors(
        final ImmutableFactorsLogConfig config)
        throws ConfigException
    {
        List<FactorGroupFactory<D>> factories = new ArrayList<>();

        for (String logFactorName: config.factors()) {
            if (!factors().containsKey(logFactorName)) {
                throw new ConfigException(
                    "Unknown log factor " + logFactorName);
            }

            FactorGroupFactory<D> factory = factors().get(logFactorName);
            if (factory instanceof LuceneFieldFactorFactory) {
                textFields.add(
                    ((LuceneFieldFactorFactory) factory).field());
            }

            factories.add(factory);
        }

        return factories;
    }

    protected T getByName(final String name, final Map<String, T> models) {
        return
            models.values().stream().filter(
                (m) -> name.equalsIgnoreCase(m.name())).findFirst().orElse(null);
    }

    @Override
    public T create(
        final CgiParams params,
        final PrefixedLogger logger,
        final UserSplit userSplit,
        final Side side,
        final Product product)
    {
        Map<String, T> experiments;
        Map<String, Map<String, T>> products =
            this.sortersFactories.get(side.name());

        if (products == null) {
            logger.warning("Side not registered in relevance " + side);
            experiments = this.sortersFactories.get(DEFAULT).get(DEFAULT);
        } else {
            experiments = products.get(product.name());
            if (experiments == null) {
                logger.warning("Product not registered in relevance " + product);
                experiments = products.get(DEFAULT);
            }
        }

        if (params.containsKey(ProxyParams.MODEL)) {
            String modelName;
            try {
                modelName = params.getString(ProxyParams.MODEL);
            } catch (BadRequestException bre) {
                return null;
            }

            logger.info("Model selected by relevance model param: " + modelName);

            T model = this.getByName(modelName, experiments);
            if (model != null) {
                return model;
            } else {
                for (Map<String, Map<String, T>> productMap :
                    this.sortersFactories.values()) {
                    for (Map<String, T> expMap : productMap.values()) {
                        model = getByName(modelName, expMap);
                        if (model != null) {
                            return model;
                        }
                    }
                }

                logger.warning("Model name " + modelName + " was not found");
            }
        }

        if (userSplit != null) {
            for (String test: userSplit.tests()) {
                T builder = experiments.get(test);
                if (builder != null) {
                    return builder;
                }
            }
        }

        return experiments.get(DEFAULT);
    }

    public Map<String, ? extends Object> status(boolean verbose) {
        return status;
    }
}
