package ru.yandex.msearch.proxy.api.async.mail.relevance;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.searcher.PlainSearcher;

import ru.yandex.msearch.proxy.ora.wmi.fields.WmiField;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.json.fieldfunction.SumMapFunction;

import ru.yandex.search.result.SearchResult;

public class MtypeRequestProcessorFactory
    implements PersonalFactorsProcessorFactory
{
    public static final MtypeRequestProcessorFactory INSTANCE =
        new MtypeRequestProcessorFactory();

    public static final String MTYPE_FREQUENCY = "user_mtype_freq";
    public static final String MTYPE_TOTAL_SHOW = "user_mtype_show";
    public static final String MTYPE_NUMBERS = "mtype_list";
    private static final String MTYPE_FIELD = "mtype_show_count";

    @Override
    public PersonalFactorsProcesor create(
        final SearchSession session,
        final User user,
        final PlainSearcher<SearchResult> searcher)
    {
        return new MtypeRequestProcessor(session, user, searcher);
    }

    private static final class MtypeRequestProcessor
        extends PersonalFactorsProcesor
    {
        private final Map<String, Long> statMap = new LinkedHashMap<>();
        private Long mtypeTotal = null;
        private volatile boolean done = false;

        public MtypeRequestProcessor(
            final SearchSession session,
            final User user,
            final PlainSearcher<SearchResult> searcher)
        {
            super(session, user, searcher);
        }

        @Override
        protected String name() {
            return "personal-mtype";
        }

        @Override
        protected BasicAsyncRequestProducerGenerator request(
            final String userRequest)
            throws BadRequestException
        {
            QueryConstructor qc = new QueryConstructor("/search?");
            qc.append("get", MTYPE_FIELD);
            qc.append("text", MTYPE_FIELD + ":*");
            qc.append("prefix", user.prefix().toString());
            qc.append("service", user.service());

            return new BasicAsyncRequestProducerGenerator(qc.toString());
        }

        private double getMtypeFrequency(
            final String types,
            final long total)
            throws JsonException
        {
            if (types != null) {
                Long docStat = statMap.get(types);
                if (docStat == null) {
                    return 0;
                } else {
                    return (1.0 * docStat) / total;
                }
            }

            return 0;
        }

        private String getMtypeList(final Document doc)
            throws JsonException
        {
            JsonList types =
                doc.envelope().getListOrNull(WmiField.TYPES);

            if (types != null) {
                List<Integer> mtypes = new ArrayList<>();
                for (JsonObject typeObj : types) {
                    mtypes.add((int) typeObj.asLong());
                }

                StringBuilder sb = new StringBuilder();
                Collections.sort(mtypes);

                for (Integer mtype : mtypes) {
                    sb.append(mtype);
                    sb.append(",");
                }

                if (sb.length() > 0) {
                    sb.setLength(sb.length() - 1);
                }

                return sb.toString();
            }

            return null;
        }

        @Override
        public synchronized void accept(
            final Document document)
            throws JsonException
        {
            if (!done) {
                return;
            }

            Map<String, String> attrs = document.doc().attrs();
            if (mtypeTotal != null) {
                String mtypes = getMtypeList(document);

                if (mtypes != null) {
                    attrs.put(MTYPE_NUMBERS, mtypes);
                }

                attrs.put(
                    MTYPE_FREQUENCY,
                    String.valueOf(
                        getMtypeFrequency(mtypes, mtypeTotal)));
                attrs.put(
                    MTYPE_TOTAL_SHOW,
                    String.valueOf(mtypeTotal));
            }
        }

        @Override
        public void completed(final SearchResult result) {
            if (result.hitsCount() == 0) {
                return;
            }

            if (result.hitsCount() > 1) {
                logger.warning(
                    "Ivalid MessageType stat for user " + result.toString());
                return;
            }


            String mtypeStat =
                result.hitsArray().get(0).attrs().get(MTYPE_FIELD);

            if (mtypeStat == null) {
                logger.warning("No " + MTYPE_FIELD
                    + " for user, skipping");
                return;
            }

            logger.info("Personal mtype search completed");

            try {
                synchronized (this) {
                    if (done) {
                        return;
                    }

                    done = true;

                    SumMapFunction.addToMap(statMap, mtypeStat);
                    if (statMap != null) {
                        mtypeTotal = statMap.get("all");
                    }

                    if (mtypeTotal != null && mtypeTotal <= 0) {
                        mtypeTotal = null;
                    }
                }
            } catch (FieldFunctionException ffe) {
                logger.log(
                    Level.WARNING,
                    "Invalid mtype format " + mtypeStat,
                    ffe);
            }
        }
    }
}