package ru.yandex.msearch.proxy.api.async.mail.relevance;


import java.util.ArrayList;
import java.util.List;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;

import ru.yandex.parser.uri.CgiParams;

public class ReceiveDateFactory extends AbstractFactorGroupFactory {
    private final ReceivedDateFactorFactory[] factories;

    public ReceiveDateFactory(
        final String name,
        final ReceivedDateFactorFactory... fs)
    {
        super(name, fs);

        this.factories = fs;
    }

    @Override
    public FactorGroup create(
        final CgiParams params,
        final PrefixedLogger logger)
    {
        List<ReceivedDateFactor> factors = new ArrayList<>(factories.length);
        double ts = System.currentTimeMillis() / 1000.0;

        for (int i = 0; i < factories.length; i++) {
            factors.add(factories[i].create(ts));
        }

        return new ReceivedDateGroup(factors);
    }

    public interface ReceivedDateFactorFactory extends FactorFactoryAttributes {
        ReceivedDateFactor create(final double ts);
    }

    private static class ReceivedDateGroup
        implements NormalizedFactorGroup<Document>
    {
        private final List<ReceivedDateFactor> factors;

        public ReceivedDateGroup(final List<ReceivedDateFactor> factors) {
            this.factors = factors;
        }

        @Override
        public void calc(
            final Document doc,
            final String request,
            final double[] res)
            throws RelevanceException, JsonException
        {
            String rcvDateStr =
                doc.doc().attrs().get(MailIndexFields.RECEIVED_DATE);

            if (rcvDateStr == null) {
                if (res != null) {
                    factors.forEach(f -> {
                        res[f.index()] = f.noField();
                    });
                }

                return;
            }

            long receiveDate;
            try {
                receiveDate = Long.parseLong(rcvDateStr);
            } catch (NumberFormatException nfe) {
                throw new RelevanceException("Invalid received date", nfe);
            }

            factors.forEach(f -> {
                if (res != null) {
                    res[f.index()] = f.calc(doc, request, receiveDate);
                }
            });
        }
    }

    public static class CommonReceivedDateFactorFactory
        implements ReceivedDateFactorFactory
    {
        private final String name;
        private final int index;
        private final boolean plain;

        public CommonReceivedDateFactorFactory(
            final String name,
            final int index,
            final boolean plain)
        {
            this.name = name;
            this.index = index;
            this.plain = plain;
        }

        public int index() {
            return index;
        }

        @Override
        public String groupName() {
            return name;
        }

        public CommonReceivedDateFactorFactory(
            final String name,
            final int index)
        {
            this(name, index, false);
        }

        @Override
        public ReceivedDateFactor create(final double ts) {
            return new LogReceivedDateFactor(ts, index);
        }
    }

    private abstract static class ReceivedDateFactor {
        private final int index;

        private ReceivedDateFactor(final int index) {
            this.index = index;
        }

        public double noField() {
            return 0;
        }

        public int index() {
            return index;
        }

        public abstract double calc(
            final Document doc,
            final String request,
            final long receivedDate);
    }

    private static class LogReceivedDateFactor
        extends ReceivedDateFactor
    {
        private final double timestamp;

        public LogReceivedDateFactor(
            final double timestamp,
            final int index)
        {
            super(index);

            this.timestamp = timestamp;
        }

        @Override
        public double calc (
            final Document doc,
            final String request,
            final long receiveDate)
        {
            double age;
            if (receiveDate > timestamp) {
                age = 0.0;
            } else {
                age = Math.log(1.0 + (timestamp - receiveDate) / 3600.0);
            }

            return age;
        }
    }
}