package ru.yandex.msearch.proxy.api.async.mail.relevance;

import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.searcher.PlainSearcher;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.request.util.SearchRequestText;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class RequestMidProcessorFactory
    implements PersonalFactorsProcessorFactory
{
    public static final RequestMidProcessorFactory INSTANCE =
        new RequestMidProcessorFactory();

    public static final String WAS_RELEVANT = "was_relevant";

    @Override
    public PersonalFactorsProcesor create(
        final SearchSession session,
        final User user,
        final PlainSearcher<SearchResult> searcher)
    {
        return new RequestMidProcessor(session, user, searcher);
    }

    public static class RequestMidProcessor
        extends PersonalFactorsProcesor
    {
        protected LinkedHashMap<String, Double> midsScores =
            new LinkedHashMap<>();

        protected volatile boolean done = false;

        public RequestMidProcessor(
            final SearchSession session,
            final User user,
            final PlainSearcher<SearchResult> searcher)
        {
            super(session, user, searcher);
        }

        @Override
        protected String name() {
            return "request-mid";
        }

        @Override
        protected boolean enabled(final String request) {
            return request != null && request.length() > 2;
        }

        @Override
        protected BasicAsyncRequestProducerGenerator request(
            final String userRequest)
            throws BadRequestException
        {
            QueryConstructor qc = new QueryConstructor("/search?");
            String escapedRequest = SearchRequestText.quoteEscape(userRequest);
            qc.append("get", MailIndexFields.REQUEST_MIDS);
            qc.append("text", "request_raw:\"" + escapedRequest + "\"");
            qc.append("prefix", user.prefix().toString());
            qc.append("service", user.service());
            return new BasicAsyncRequestProducerGenerator(qc.toString());
        }

        @Override
        public synchronized void accept(
            final Document document)
            throws JsonException
        {
            if (!done) {
                return;
            }

            Map<String, String> attrs = document.doc().attrs();
            String mid = attrs.get(MailIndexFields.MID);
            if (mid == null) {
                return;
            }

            Double weight = midsScores.get(mid);
            if (weight != null) {
                attrs.put(WAS_RELEVANT, weight.toString());
            }
        }

        protected double processRequestData(final SearchDocument document) {
            return 1.0;
        }

        protected int proccessRequestResult(final SearchResult result) {
            int added = 0;
            for (SearchDocument doc : result.hitsArray()) {
                String mid =
                    doc.attrs().getOrDefault(
                        MailIndexFields.REQUEST_MIDS,
                        null);

                if (mid != null) {
                    double weight = processRequestData(doc);
                    if (weight > 0) {
                        midsScores.put(mid, weight);
                        added++;
                    }
                }
            }

            return added;
        }

        @Override
        public void completed(final SearchResult result) {
            if (result.hitsCount() <= 0 || result.hitsArray().size() <= 0) {
                logger.info("No mid stored, for request");
                return;
            }

            logger.info(
                "Personal factors search completed "
                    + result.hitsCount());
            int added;
            synchronized (this) {
                if (done) {
                    return;
                }

                done = true;

                added = proccessRequestResult(result);
            }

            if (added <= 0) {
                logger.info("No mid stored, for request");
            }
        }
    }
}
