package ru.yandex.msearch.proxy.api.async.mail.relevance.search;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;

import org.apache.http.Header;

import ru.yandex.http.util.YandexHeaders;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.relevance.MtypeRequestProcessorFactory;
import ru.yandex.msearch.proxy.api.async.mail.relevance.RelevanceException;
import ru.yandex.msearch.proxy.api.async.mail.rules.PersonalFactorsGatherRule;

import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;

import ru.yandex.msearch.proxy.logger.ProxyTskvLogger;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.tskv.TskvString;

public class LoggingSession extends RankingSession {
    public static final String X_REQUEST_ID = "x_request_id";
    public static final String REQUEST_ID = "reqid";
    public static final String REQUEST = "request_text";
    public static final String MID = "mid";
    public static final String POSITION = "pos";
    public static final String NON_RANKED_POSITION = "norank_pos";
    public static final String PRODUCT = "product";
    public static final String SIDE = "side";
    public static final String UNIXTIME = "unixtime";
    public static final String EXPERIMENTS = "exps";
    public static final String RANK_MODEL = "model";
    public static final String MTYPES = "mtypes";
    public static final String SCORE = "score";

    private final long timestamp = System.currentTimeMillis() / 1000;

    public LoggingSession(
        final SearchSession session,
        final AbstractMailSearchRelevanceSorter sorter)
    {
        super(session, sorter);
    }

    private String extractRequestId(final RequestInfo requestInfo) {
        Header header = requestInfo.request()
            .getLastHeader(YandexHeaders.X_REQUEST_ID);
        if (header == null) {
            return null;
        }

        return header.getValue();
    }

    protected ProxyTskvLogger prepareLogger(
        final SearchSession session,
        final PrefixedLogger logger)
        throws IOException
    {
        CgiParams params = session.params();
        RequestInfo requestInfo = session.requestInfo();

        ProxyTskvLogger tskvLogger = sorter.factorsLogger();

        if (tskvLogger == null) {
            return null;
        }

        String xRequestId = extractRequestId(requestInfo);
        String reqId = params.getOrNull("reqid");
        MailSearchOptions options = requestInfo.options();
        String request = options.request();

        if (request == null) {
            logger.info("factors logging disabled: request null");
            return null;
        }

        if (xRequestId == null && reqId == null) {
            logger.info("factors logging disabled: xrequest and reqid null");
            return null;
        }

        TskvString prefix = tskvLogger.record(requestInfo.session().context());

        if (xRequestId != null) {
            prefix.append(X_REQUEST_ID, xRequestId);
        }

        if (reqId != null) {
            prefix.append(REQUEST_ID, reqId);
        }

        prefix.append(REQUEST, request);
        prefix.append(UNIXTIME, timestamp);
        if (options.userSplit() != null) {
            prefix.append(EXPERIMENTS, options.userSplit().testsToString());
        } else {
            prefix.append(EXPERIMENTS, "");
        }

        return tskvLogger.prefix(prefix);
    }

    @Override
    protected Documents rank(final Documents documents) {
        ProxyTskvLogger factorsLogger;
        try {
            factorsLogger = prepareLogger(session, logger);
        } catch (IOException ioe) {
            logger.log(Level.WARNING, "Failed prepare factors logger", ioe);
            return documents;
        }

        if (factorsLogger == null) {
            logger.info("Logging factors disabled");
            return super.rank(documents);
        }

        RankingDocuments result;
        try {
            result = rankDocuments(documents);
        } catch (JsonException | RelevanceException | NumberFormatException e) {
            logger.log(Level.WARNING, "Unable to rank documents", e);
            return documents;
        }

        RequestInfo requestInfo = session.requestInfo();

        int threshold =
            Math.min(
                result.size(),
                requestInfo.offset() + requestInfo.length());

        for (int pos = requestInfo.offset(); pos < threshold; pos++) {
            RankedDocument rDoc = result.documents().get(pos);

            TskvString record = factorsLogger.record();
            //if (pos < result.topRelevantSize()) {
            if (!sorter.topRelevant()
                && pos < sorter.rankedPositions())
            {
                record.append(RANK_MODEL, sorter.name());
            } else if (sorter.topRelevant() && pos < result.relevant())
            {
                record.append(RANK_MODEL, sorter.name());
            } else {
                record.append(RANK_MODEL, "");
            }

            record.append(POSITION, pos + 1);
            record.append(MID, rDoc.id());
            record.append(
                MTYPES,
                rDoc.best().doc().attrs()
                    .getOrDefault(MtypeRequestProcessorFactory.MTYPE_NUMBERS, ""));

            record.append(
                PRODUCT,
                String.valueOf(session.requestInfo().options().product())
                    .toLowerCase(Locale.ROOT));

            record.append(
                SIDE,
                String.valueOf(session.requestInfo().options().side())
                    .toLowerCase(Locale.ROOT));

            record.append(SCORE, rDoc.score());

            record.append(NON_RANKED_POSITION, rDoc.nonRankedPos());

            double[] factors = rDoc.factors();
            int[] logIndexes = sorter.logIndexes();
            List<String> names = sorter.names();

            for (int j = 0; j < logIndexes.length; j++) {
                record.append(
                    names.get(logIndexes[j]),
                    factors[logIndexes[j]]);
            }

            factorsLogger.log(Level.INFO, record);
        }

        return result;
    }
}
