package ru.yandex.msearch.proxy.api.async.mail.relevance.search;

import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Level;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.matrixnet.MatrixnetModel;

import ru.yandex.msearch.proxy.api.async.mail.DefaultSearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;

import ru.yandex.msearch.proxy.api.async.mail.relevance.RelevanceException;

import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchModelSorterFactory.MailSearchFactorsContext;

import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.parser.uri.CgiParams;

public class ModelMailSearchRelevanceSorter
    extends AbstractMailSearchRelevanceSorter
{
    // num documents to be ranked by model
    private final MailSearchFactorsContext context;
    private final int rankLength;

    public ModelMailSearchRelevanceSorter(
        final MailSearchRelevanceFactory relevanceFactory,
        final MailSearchFactorsContext context,
        final CgiParams params,
        final PrefixedLogger logger,
        final int rankLength)
    {
        super(relevanceFactory, context, params, logger);

        this.context = context;

        this.rankLength = rankLength;
    }

    public ModelMailSearchRelevanceSorter(
        final MailSearchRelevanceFactory relevanceFactory,
        final MailSearchFactorsContext context,
        final SearchSession session)
    {
        this(
            relevanceFactory,
            context,
            session.params(),
            session.httpSession().logger(),
            totalRankabeDocs(session, relevanceFactory.proxyConfig()));
    }

    private static int totalRankabeDocs(
        final SearchSession session,
        final ImmutableMsearchProxyConfig config) {
        int multiplier = 1;
        int docsToRank = -1;
        RequestInfo requestInfo = session.requestInfo();

        try {
            docsToRank = session.params().getInt("rank-size", -1);
            multiplier = new DefaultSearchAttributes(
                session.params()).requestedResultsMultipier();
        } catch (BadRequestException bre) {
            session.httpSession().logger()
                .log(Level.WARNING, "Unable calculate docs in ranking", bre);
        }

        if (docsToRank < 0) {
            docsToRank = Math.max(
                config.filterSearchConfig().batchSize(),
                requestInfo.length() * multiplier);
        }

        session.httpSession().logger().info("Maximum docs to rank " + docsToRank);
        return docsToRank;
    }

    public MatrixnetModel model() {
        return context.model();
    }

    public String name() {
        return context.model().name();
    }

    public List<RankedDocument> sort(
        final List<? extends DocumentsGroup> docs,
        final String request)
        throws RelevanceException, JsonException
    {
        List<RankedDocument> rankedDocs = prepare(docs, request);

        for (RankedDocument sD: rankedDocs) {
            rank(sD, request);
        }

        RankedDocument[] rdArray = new RankedDocument[rankedDocs.size()];
        rankedDocs.toArray(rdArray);
        int docsInRank = Math.min(rankLength, rankedDocs.size());

        Arrays.sort(rdArray, 0 , docsInRank, RankedDocument.SCORE_COMPARATOR);

        ListIterator<RankedDocument> i = rankedDocs.listIterator();
        for (RankedDocument rd: rdArray) {
            i.next();
            i.set(rd);
        }

        return rankedDocs;
    }

    public void rank(final RankedDocument doc, final String request)
        throws RelevanceException, JsonException
    {
        double[] values = calculate(doc.best(), request);
        doc.score(context.model().score(values));
        doc.factors(values);
    }

    @Override
    public int minSerpSize() {
        return context.config().minSerpSizeRanking();
    }

    @Override
    public int rankedPositions() {
        return context.config().rankedPositions();
    }
}
